/*
This file is part of mfaktc.
Copyright (C) 2012  George Woltman (woltman@alum.mit.edu)

mfaktc is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mfaktc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with mfaktc.  If not, see <http://www.gnu.org/licenses/>.
*/

/* 224 - 416 - 448-bit extensions are built from tf_192.h routines; S.Batalov 2012 */

__device__ static int cmp_ge_224(int224 a, int224 b)
/* checks if a is greater or equal than b */
{
  if(a.d6 != b.d6) return(a.d6 > b.d6);
  if(a.d5 != b.d5) return(a.d5 > b.d5);
  if(a.d4 != b.d4) return(a.d4 > b.d4);
  if(a.d3 != b.d3) return(a.d3 > b.d3);
  if(a.d2 != b.d2) return(a.d2 > b.d2);
  if(a.d1 != b.d1) return(a.d1 > b.d1);
  return(a.d0 >= b.d0);
}


__device__ static void sub_224(int224 *res, int224 a, int224 b)
/* a must be greater or equal b!
res = a - b */
{
  res->d0 = __sub_cc (a.d0, b.d0);
  res->d1 = __subc_cc(a.d1, b.d1);
  res->d2 = __subc_cc(a.d2, b.d2);
  res->d3 = __subc_cc(a.d3, b.d3);
  res->d4 = __subc_cc(a.d4, b.d4);
  res->d5 = __subc_cc(a.d5, b.d5);
  res->d6 = __subc   (a.d6, b.d6);
}


__device__ static void square_224_448(int448 *res, int224 a)
/* res = a^2, assuming that a is < 2^223 (a.d6 < 2^31)! */
{
  asm("{\n\t"
      "mul.lo.u32      %1, %14, %15;\n\t"     /* (a.d0 * a.d1).lo */
      "mul.lo.u32      %2, %14, %16;\n\t"     /* (a.d0 * a.d2).lo */
      "mul.lo.u32      %3, %14, %17;\n\t"     /* (a.d0 * a.d3).lo */
      "mul.lo.u32      %4, %14, %18;\n\t"     /* (a.d0 * a.d4).lo */
      "mul.lo.u32      %5, %14, %19;\n\t"     /* (a.d0 * a.d5).lo */
      "mul.lo.u32      %6, %14, %20;\n\t"     /* (a.d0 * a.d6).lo */

      "mad.hi.cc.u32   %2, %14, %15, %2;\n\t" /* (a.d0 * a.d1).hi */
      "madc.hi.cc.u32  %3, %14, %16, %3;\n\t" /* (a.d0 * a.d2).hi */
      "madc.hi.cc.u32  %4, %14, %17, %4;\n\t" /* (a.d0 * a.d3).hi */
      "madc.hi.cc.u32  %5, %14, %18, %5;\n\t" /* (a.d0 * a.d4).hi */
      "madc.hi.cc.u32  %6, %14, %19, %6;\n\t" /* (a.d0 * a.d5).hi */
      "madc.hi.u32     %7, %14, %20, 0;\n\t"  /* (a.d0 * a.d6).hi */

      "mad.lo.cc.u32   %3, %15, %16, %3;\n\t" /* (a.d1 * a.d2).lo */
      "madc.hi.cc.u32  %4, %15, %16, %4;\n\t" /* (a.d1 * a.d2).hi */
      "madc.hi.cc.u32  %5, %15, %17, %5;\n\t" /* (a.d1 * a.d3).hi */
      "madc.hi.cc.u32  %6, %15, %18, %6;\n\t" /* (a.d1 * a.d4).hi */
      "madc.hi.cc.u32  %7, %15, %19, %7;\n\t" /* (a.d1 * a.d5).hi */
      "madc.hi.u32     %8, %15, %20, 0;\n\t"  /* (a.d1 * a.d6).hi */

      "mad.lo.cc.u32   %4, %15, %17, %4;\n\t" /* (a.d1 * a.d3).lo */
      "madc.lo.cc.u32  %5, %15, %18, %5;\n\t" /* (a.d1 * a.d4).lo */
      "madc.lo.cc.u32  %6, %15, %19, %6;\n\t" /* (a.d1 * a.d5).lo */
      "madc.lo.cc.u32  %7, %15, %20, %7;\n\t" /* (a.d1 * a.d6).lo */
      "madc.hi.cc.u32  %8, %16, %19, %8;\n\t" /* (a.d2 * a.d5).hi */
      "madc.hi.u32     %9, %16, %20, 0;\n\t"  /* (a.d2 * a.d6).hi */

      "mad.lo.cc.u32   %5, %16, %17, %5;\n\t" /* (a.d2 * a.d3).lo */
      "madc.hi.cc.u32  %6, %16, %17, %6;\n\t" /* (a.d2 * a.d3).hi */
      "madc.hi.cc.u32  %7, %16, %18, %7;\n\t" /* (a.d2 * a.d4).hi */
      "madc.hi.cc.u32  %8, %17, %18, %8;\n\t" /* (a.d3 * a.d4).hi */
      "madc.hi.cc.u32  %9, %17, %19, %9;\n\t" /* (a.d3 * a.d5).hi */
      "madc.hi.u32     %10,%17, %20, 0;\n\t"  /* (a.d3 * a.d6).hi */

      "mad.lo.cc.u32   %6, %16, %18, %6;\n\t"  /* (a.d2 * a.d4).lo */
      "madc.lo.cc.u32  %7, %16, %19, %7;\n\t"  /* (a.d2 * a.d5).lo */
      "madc.lo.cc.u32  %8, %16, %20, %8;\n\t"  /* (a.d2 * a.d6).lo */
      "madc.lo.cc.u32  %9, %17, %20, %9;\n\t"  /* (a.d3 * a.d6).lo */
      "madc.hi.cc.u32  %10,%18, %19, %10;\n\t" /* (a.d4 * a.d5).hi */
      "madc.hi.u32     %11,%18, %20, 0;\n\t"   /* (a.d4 * a.d6).hi */

      "mad.lo.cc.u32   %7, %17, %18, %7;\n\t"  /* (a.d3 * a.d4).lo */
      "madc.lo.cc.u32  %8, %17, %19, %8;\n\t"  /* (a.d3 * a.d5).lo */
      "madc.lo.cc.u32  %9, %18, %19, %9;\n\t"  /* (a.d4 * a.d5).lo */
      "madc.lo.cc.u32  %10,%18, %20, %10;\n\t" /* (a.d4 * a.d6).lo */
      "madc.lo.cc.u32  %11,%19, %20, %11;\n\t" /* (a.d5 * a.d6).lo */
      "madc.hi.u32     %12,%19, %20, 0;\n\t"   /* (a.d5 * a.d6).hi */

      "add.cc.u32      %1, %1, %1;\n\t"       /* Double the partial results */
      "addc.cc.u32     %2, %2, %2;\n\t"
      "addc.cc.u32     %3, %3, %3;\n\t"
      "addc.cc.u32     %4, %4, %4;\n\t"
      "addc.cc.u32     %5, %5, %5;\n\t"
      "addc.cc.u32     %6, %6, %6;\n\t"
      "addc.cc.u32     %7, %7, %7;\n\t"
      "addc.cc.u32     %8, %8, %8;\n\t"
      "addc.cc.u32     %9, %9, %9;\n\t"
      "addc.cc.u32     %10, %10, %10;\n\t"
      "addc.cc.u32     %11, %11, %11;\n\t"
      "addc.u32        %12, %12, %12;\n\t"

      "mul.lo.u32      %0, %14, %14;\n\t"       /* (a.d0 * a.d0).lo */
      "mad.hi.cc.u32   %1, %14, %14, %1;\n\t"   /* (a.d0 * a.d0).hi */
      "madc.lo.cc.u32  %2, %15, %15, %2;\n\t"   /* (a.d1 * a.d1).lo */
      "madc.hi.cc.u32  %3, %15, %15, %3;\n\t"   /* (a.d1 * a.d1).hi */
      "madc.lo.cc.u32  %4, %16, %16, %4;\n\t"   /* (a.d2 * a.d2).lo */
      "madc.hi.cc.u32  %5, %16, %16, %5;\n\t"   /* (a.d2 * a.d2).hi */
      "madc.lo.cc.u32  %6, %17, %17, %6;\n\t"   /* (a.d3 * a.d3).lo */
      "madc.hi.cc.u32  %7, %17, %17, %7;\n\t"   /* (a.d3 * a.d3).hi */
      "madc.lo.cc.u32  %8, %18, %18, %8;\n\t"   /* (a.d4 * a.d4).lo */
      "madc.hi.cc.u32  %9, %18, %18, %9;\n\t"   /* (a.d4 * a.d4).hi */
      "madc.lo.cc.u32  %10, %19, %19, %10;\n\t" /* (a.d5 * a.d5).lo */
      "madc.hi.cc.u32  %11, %19, %19, %11;\n\t" /* (a.d5 * a.d5).hi */
      "madc.lo.cc.u32  %12, %20, %20, %12;\n\t" /* (a.d6 * a.d6).lo */
      "madc.hi.u32     %13, %20, %20, 0;\n\t"   /* (a.d6 * a.d6).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5),
        "=r" (res->d6), "=r" (res->d7), "=r" (res->d8), "=r" (res->d9), "=r" (res->d10), "=r" (res->d11), 
        "=r" (res->d12), "=r" (res->d13)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6));
}


__device__ static void square_224_416(int416 *res, int224 a)
/* res = a^2, assuming that a is < 2^208 (a.d5 < 2^16)! */
{
  asm("{\n\t"
      "mul.lo.u32      %1, %13, %14;\n\t"     /* (a.d0 * a.d1).lo */
      "mul.lo.u32      %2, %13, %15;\n\t"     /* (a.d0 * a.d2).lo */
      "mul.lo.u32      %3, %13, %16;\n\t"     /* (a.d0 * a.d3).lo */
      "mul.lo.u32      %4, %13, %17;\n\t"     /* (a.d0 * a.d4).lo */
      "mul.lo.u32      %5, %13, %18;\n\t"     /* (a.d0 * a.d5).lo */
      "mul.lo.u32      %6, %13, %19;\n\t"     /* (a.d0 * a.d6).lo */

      "mad.hi.cc.u32   %2, %13, %14, %2;\n\t" /* (a.d0 * a.d1).hi */
      "madc.hi.cc.u32  %3, %13, %15, %3;\n\t" /* (a.d0 * a.d2).hi */
      "madc.hi.cc.u32  %4, %13, %16, %4;\n\t" /* (a.d0 * a.d3).hi */
      "madc.hi.cc.u32  %5, %13, %17, %5;\n\t" /* (a.d0 * a.d4).hi */
      "madc.hi.cc.u32  %6, %13, %18, %6;\n\t" /* (a.d0 * a.d5).hi */
      "madc.hi.u32     %7, %13, %19, 0;\n\t"  /* (a.d0 * a.d6).hi */

      "mad.lo.cc.u32   %3, %14, %15, %3;\n\t" /* (a.d1 * a.d2).lo */
      "madc.hi.cc.u32  %4, %14, %15, %4;\n\t" /* (a.d1 * a.d2).hi */
      "madc.hi.cc.u32  %5, %14, %16, %5;\n\t" /* (a.d1 * a.d3).hi */
      "madc.hi.cc.u32  %6, %14, %17, %6;\n\t" /* (a.d1 * a.d4).hi */
      "madc.hi.cc.u32  %7, %14, %18, %7;\n\t" /* (a.d1 * a.d5).hi */
      "madc.hi.u32     %8, %14, %19, 0;\n\t"  /* (a.d1 * a.d6).hi */

      "mad.lo.cc.u32   %4, %14, %16, %4;\n\t" /* (a.d1 * a.d3).lo */
      "madc.lo.cc.u32  %5, %14, %17, %5;\n\t" /* (a.d1 * a.d4).lo */
      "madc.lo.cc.u32  %6, %14, %18, %6;\n\t" /* (a.d1 * a.d5).lo */
      "madc.lo.cc.u32  %7, %14, %19, %7;\n\t" /* (a.d1 * a.d6).lo */
      "madc.hi.cc.u32  %8, %15, %18, %8;\n\t" /* (a.d2 * a.d5).hi */
      "madc.hi.u32     %9, %15, %19, 0;\n\t"  /* (a.d2 * a.d6).hi */

      "mad.lo.cc.u32   %5, %15, %16, %5;\n\t" /* (a.d2 * a.d3).lo */
      "madc.hi.cc.u32  %6, %15, %16, %6;\n\t" /* (a.d2 * a.d3).hi */
      "madc.hi.cc.u32  %7, %15, %17, %7;\n\t" /* (a.d2 * a.d4).hi */
      "madc.hi.cc.u32  %8, %16, %17, %8;\n\t" /* (a.d3 * a.d4).hi */
      "madc.hi.cc.u32  %9, %16, %18, %9;\n\t" /* (a.d3 * a.d5).hi */
      "madc.hi.u32     %10,%16, %19, 0;\n\t"  /* (a.d3 * a.d6).hi */

      "mad.lo.cc.u32   %6, %15, %17, %6;\n\t"  /* (a.d2 * a.d4).lo */
      "madc.lo.cc.u32  %7, %15, %18, %7;\n\t"  /* (a.d2 * a.d5).lo */
      "madc.lo.cc.u32  %8, %15, %19, %8;\n\t"  /* (a.d2 * a.d6).lo */
      "madc.lo.cc.u32  %9, %16, %19, %9;\n\t"  /* (a.d3 * a.d6).lo */
      "madc.hi.cc.u32  %10,%17, %18, %10;\n\t" /* (a.d4 * a.d5).hi */
      "madc.hi.u32     %11,%17, %19, 0;\n\t"   /* (a.d4 * a.d6).hi */

      "mad.lo.cc.u32   %7, %16, %17, %7;\n\t"  /* (a.d3 * a.d4).lo */
      "madc.lo.cc.u32  %8, %16, %18, %8;\n\t"  /* (a.d3 * a.d5).lo */
      "madc.lo.cc.u32  %9, %17, %18, %9;\n\t"  /* (a.d4 * a.d5).lo */
      "madc.lo.cc.u32  %10,%17, %19, %10;\n\t" /* (a.d4 * a.d6).lo */
      "madc.lo.cc.u32  %11,%18, %19, %11;\n\t" /* (a.d5 * a.d6).lo */
      "madc.hi.u32     %12, %18, %19, 0;\n\t"  /* (a.d5 * a.d6).hi */

      "add.cc.u32      %1, %1, %1;\n\t"       /* Double the partial results */
      "addc.cc.u32     %2, %2, %2;\n\t"
      "addc.cc.u32     %3, %3, %3;\n\t"
      "addc.cc.u32     %4, %4, %4;\n\t"
      "addc.cc.u32     %5, %5, %5;\n\t"
      "addc.cc.u32     %6, %6, %6;\n\t"
      "addc.cc.u32     %7, %7, %7;\n\t"
      "addc.cc.u32     %8, %8, %8;\n\t"
      "addc.cc.u32     %9, %9, %9;\n\t"
      "addc.cc.u32     %10, %10, %10;\n\t"
      "addc.cc.u32     %11, %11, %11;\n\t"
      "addc.u32        %12, %12, %12;\n\t"

      "mul.lo.u32      %0, %13, %13;\n\t"       /* (a.d0 * a.d0).lo */
      "mad.hi.cc.u32   %1, %13, %13, %1;\n\t"   /* (a.d0 * a.d0).hi */
      "madc.lo.cc.u32  %2, %14, %14, %2;\n\t"   /* (a.d1 * a.d1).lo */
      "madc.hi.cc.u32  %3, %14, %14, %3;\n\t"   /* (a.d1 * a.d1).hi */
      "madc.lo.cc.u32  %4, %15, %15, %4;\n\t"   /* (a.d2 * a.d2).lo */
      "madc.hi.cc.u32  %5, %15, %15, %5;\n\t"   /* (a.d2 * a.d2).hi */
      "madc.lo.cc.u32  %6, %16, %16, %6;\n\t"   /* (a.d3 * a.d3).lo */
      "madc.hi.cc.u32  %7, %16, %16, %7;\n\t"   /* (a.d3 * a.d3).hi */
      "madc.lo.cc.u32  %8, %17, %17, %8;\n\t"   /* (a.d4 * a.d4).lo */
      "madc.hi.cc.u32  %9, %17, %17, %9;\n\t"   /* (a.d4 * a.d4).hi */
      "madc.lo.cc.u32  %10, %18, %18, %10;\n\t" /* (a.d5 * a.d5).lo */
      "madc.hi.cc.u32  %11, %18, %18, %11;\n\t" /* (a.d5 * a.d5).hi */
      "madc.lo.u32     %12, %19, %19, %12;\n\t" /* (a.d6 * a.d6).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5),
        "=r" (res->d6), "=r" (res->d7), "=r" (res->d8), "=r" (res->d9), "=r" (res->d10), "=r" (res->d11), 
        "=r" (res->d12)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6));
}


__device__ static void mul_224_448_no_low7(int224 *res, int224 a, int224 b)
/*
res ~= a * b / 2^224
Carries into res.d0 are NOT computed. So the result differs from a full mul_224_448() / 2^224.
In a full mul_224_448() there are ten possible carries from res.d5 to res.d6. So ignoring the carries
the result is 0 to 10 lower than a full mul_224_448() / 2^224.
*/
{
  asm("{\n\t"
      "mul.hi.u32      %0, %7, %20;\n\t"     /* (a.d0 * b.d6).hi */

      "mad.lo.cc.u32   %0, %8, %20, %0;\n\t" /* (a.d1 * b.d6).lo */
      "addc.u32        %1, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %8, %19, %0;\n\t" /* (a.d1 * b.d5).hi */
      "madc.hi.u32     %1, %8, %20, %1;\n\t" /* (a.d1 * b.d6).hi */

      "mad.lo.cc.u32   %0, %9, %19, %0;\n\t" /* (a.d2 * b.d5).lo */
      "madc.lo.cc.u32  %1, %9, %20, %1;\n\t" /* (a.d2 * b.d6).lo */
      "addc.u32        %2, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %9, %18, %0;\n\t" /* (a.d2 * b.d4).hi */
      "madc.hi.cc.u32  %1, %9, %19, %1;\n\t" /* (a.d2 * b.d5).hi */
      "madc.hi.u32     %2, %9, %20, %2;\n\t" /* (a.d2 * b.d6).hi */

      "mad.lo.cc.u32   %0, %10, %18, %0;\n\t" /* (a.d3 * b.d4).lo */
      "madc.lo.cc.u32  %1, %10, %19, %1;\n\t" /* (a.d3 * b.d5).lo */
      "madc.lo.cc.u32  %2, %10, %20, %2;\n\t" /* (a.d3 * b.d6).lo */
      "addc.u32        %3, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %10, %17, %0;\n\t" /* (a.d3 * b.d3).hi */
      "madc.hi.cc.u32  %1, %10, %18, %1;\n\t" /* (a.d3 * b.d4).hi */
      "madc.hi.cc.u32  %2, %10, %19, %2;\n\t" /* (a.d3 * b.d5).hi */
      "madc.hi.u32     %3, %10, %20, %3;\n\t" /* (a.d3 * b.d6).hi */

      "mad.lo.cc.u32   %0, %11, %17, %0;\n\t" /* (a.d4 * b.d3).lo */
      "madc.lo.cc.u32  %1, %11, %18, %1;\n\t" /* (a.d4 * b.d4).lo */
      "madc.lo.cc.u32  %2, %11, %19, %2;\n\t" /* (a.d4 * b.d5).lo */
      "madc.lo.cc.u32  %3, %11, %20, %3;\n\t" /* (a.d4 * b.d6).lo */
      "addc.u32        %4, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %11, %16, %0;\n\t" /* (a.d4 * b.d2).hi */
      "madc.hi.cc.u32  %1, %11, %17, %1;\n\t" /* (a.d4 * b.d3).hi */
      "madc.hi.cc.u32  %2, %11, %18, %2;\n\t" /* (a.d4 * b.d4).hi */
      "madc.hi.cc.u32  %3, %11, %19, %3;\n\t" /* (a.d4 * b.d5).hi */
      "madc.hi.u32     %4, %11, %20, %4;\n\t" /* (a.d4 * b.d6).hi */

      "mad.lo.cc.u32   %0, %12, %16, %0;\n\t" /* (a.d5 * b.d2).lo */
      "madc.lo.cc.u32  %1, %12, %17, %1;\n\t" /* (a.d5 * b.d3).lo */
      "madc.lo.cc.u32  %2, %12, %18, %2;\n\t" /* (a.d5 * b.d4).lo */
      "madc.lo.cc.u32  %3, %12, %19, %3;\n\t" /* (a.d5 * b.d5).lo */
      "madc.lo.cc.u32  %4, %12, %20, %4;\n\t" /* (a.d5 * b.d6).lo */
      "addc.u32        %5, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %12, %15, %0;\n\t" /* (a.d5 * b.d1).hi */
      "madc.hi.cc.u32  %1, %12, %16, %1;\n\t" /* (a.d5 * b.d2).hi */
      "madc.hi.cc.u32  %2, %12, %17, %2;\n\t" /* (a.d5 * b.d3).hi */
      "madc.hi.cc.u32  %3, %12, %18, %3;\n\t" /* (a.d5 * b.d4).hi */
      "madc.hi.cc.u32  %4, %12, %19, %4;\n\t" /* (a.d5 * b.d5).hi */
      "madc.hi.u32     %5, %12, %20, %5;\n\t" /* (a.d5 * b.d6).hi */

      "mad.lo.cc.u32   %0, %13, %15, %0;\n\t" /* (a.d6 * b.d1).lo */
      "madc.lo.cc.u32  %1, %13, %16, %1;\n\t" /* (a.d6 * b.d2).lo */
      "madc.lo.cc.u32  %2, %13, %17, %2;\n\t" /* (a.d6 * b.d3).lo */
      "madc.lo.cc.u32  %3, %13, %18, %3;\n\t" /* (a.d6 * b.d4).lo */
      "madc.lo.cc.u32  %4, %13, %19, %4;\n\t" /* (a.d6 * b.d5).lo */
      "madc.lo.cc.u32  %5, %13, %20, %5;\n\t" /* (a.d6 * b.d6).lo */
      "addc.u32        %6, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %13, %14, %0;\n\t" /* (a.d6 * b.d0).hi */
      "madc.hi.cc.u32  %1, %13, %15, %1;\n\t" /* (a.d6 * b.d1).hi */
      "madc.hi.cc.u32  %2, %13, %16, %2;\n\t" /* (a.d6 * b.d2).hi */
      "madc.hi.cc.u32  %3, %13, %17, %3;\n\t" /* (a.d6 * b.d3).hi */
      "madc.hi.cc.u32  %4, %13, %18, %4;\n\t" /* (a.d6 * b.d4).hi */
      "madc.hi.cc.u32  %5, %13, %19, %5;\n\t" /* (a.d6 * b.d5).hi */
      "madc.hi.u32     %6, %13, %20, %6;\n\t" /* (a.d6 * b.d6).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5), "=r" (res->d6)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6),
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3), "r" (b.d4), "r" (b.d5), "r" (b.d6));
}


__device__ static void mul_224(int224 *res, int224 a, int224 b)
/* res = a * b (only lower 224 bits of the result) */
{
  asm("{\n\t"
      "mul.lo.u32      %0, %7, %14;\n\t"     /* (a.d0 * b.d0).lo */
      "mul.lo.u32      %1, %7, %15;\n\t"     /* (a.d0 * b.d1).lo */
      "mul.lo.u32      %2, %7, %16;\n\t"     /* (a.d0 * b.d2).lo */
      "mul.lo.u32      %3, %7, %17;\n\t"     /* (a.d0 * b.d3).lo */
      "mul.lo.u32      %4, %7, %18;\n\t"     /* (a.d0 * b.d4).lo */
      "mul.lo.u32      %5, %7, %19;\n\t"     /* (a.d0 * b.d5).lo */
      "mul.lo.u32      %6, %7, %20;\n\t"     /* (a.d0 * b.d6).lo */

      "mad.hi.cc.u32   %1, %7, %14, %1;\n\t" /* (a.d0 * b.d0).hi */
      "madc.hi.cc.u32  %2, %7, %15, %2;\n\t" /* (a.d0 * b.d1).hi */
      "madc.hi.cc.u32  %3, %7, %16, %3;\n\t" /* (a.d0 * b.d2).hi */
      "madc.hi.cc.u32  %4, %7, %17, %4;\n\t" /* (a.d0 * b.d3).hi */
      "madc.hi.cc.u32  %5, %7, %18, %5;\n\t" /* (a.d0 * b.d4).hi */
      "madc.hi.u32     %6, %7, %19, %6;\n\t" /* (a.d0 * b.d5).hi */

      "mad.lo.cc.u32   %1, %8, %14, %1;\n\t" /* (a.d1 * b.d0).lo */
      "madc.lo.cc.u32  %2, %8, %15, %2;\n\t" /* (a.d1 * b.d1).lo */
      "madc.lo.cc.u32  %3, %8, %16, %3;\n\t" /* (a.d1 * b.d2).lo */
      "madc.lo.cc.u32  %4, %8, %17, %4;\n\t" /* (a.d1 * b.d3).lo */
      "madc.lo.cc.u32  %5, %8, %18, %5;\n\t" /* (a.d1 * b.d4).lo */
      "madc.lo.u32     %6, %8, %19, %6;\n\t" /* (a.d1 * b.d5).lo */

      "mad.hi.cc.u32   %2, %8, %14, %2;\n\t" /* (a.d1 * b.d0).hi */
      "madc.hi.cc.u32  %3, %8, %15, %3;\n\t" /* (a.d1 * b.d1).hi */
      "madc.hi.cc.u32  %4, %8, %16, %4;\n\t" /* (a.d1 * b.d2).hi */
      "madc.hi.cc.u32  %5, %8, %17, %5;\n\t" /* (a.d1 * b.d3).hi */
      "madc.hi.u32     %6, %8, %18, %6;\n\t" /* (a.d1 * b.d4).hi */

      "mad.lo.cc.u32   %2, %9, %14, %2;\n\t" /* (a.d2 * b.d0).lo */
      "madc.lo.cc.u32  %3, %9, %15, %3;\n\t" /* (a.d2 * b.d1).lo */
      "madc.lo.cc.u32  %4, %9, %16, %4;\n\t" /* (a.d2 * b.d2).lo */
      "madc.lo.cc.u32  %5, %9, %17, %5;\n\t" /* (a.d2 * b.d3).lo */
      "madc.lo.u32     %6, %9, %18, %6;\n\t" /* (a.d2 * b.d4).lo */

      "mad.hi.cc.u32   %3, %9, %14, %3;\n\t" /* (a.d2 * b.d0).hi */
      "madc.hi.cc.u32  %4, %9, %15, %4;\n\t" /* (a.d2 * b.d1).hi */
      "madc.hi.cc.u32  %5, %9, %16, %5;\n\t" /* (a.d2 * b.d2).hi */
      "madc.hi.u32     %6, %9, %17, %6;\n\t" /* (a.d2 * b.d3).hi */

      "mad.lo.cc.u32   %3, %10, %14, %3;\n\t" /* (a.d3 * b.d0).lo */
      "madc.lo.cc.u32  %4, %10, %15, %4;\n\t" /* (a.d3 * b.d1).lo */
      "madc.lo.cc.u32  %5, %10, %16, %5;\n\t" /* (a.d3 * b.d2).lo */
      "madc.lo.u32     %6, %10, %17, %6;\n\t" /* (a.d3 * b.d3).lo */

      "mad.hi.cc.u32   %4, %10, %14, %4;\n\t" /* (a.d3 * b.d0).hi */
      "madc.hi.cc.u32  %5, %10, %15, %5;\n\t" /* (a.d3 * b.d1).hi */
      "madc.hi.u32     %6, %10, %16, %6;\n\t" /* (a.d3 * b.d2).hi */

      "mad.lo.cc.u32   %4, %11, %14, %4;\n\t" /* (a.d4 * b.d0).lo */
      "madc.lo.cc.u32  %5, %11, %15, %5;\n\t" /* (a.d4 * b.d1).lo */
      "madc.lo.u32     %6, %11, %16, %6;\n\t" /* (a.d4 * b.d2).lo */

      "mad.hi.cc.u32   %5, %11, %14, %5;\n\t" /* (a.d4 * b.d0).hi */
      "madc.hi.u32     %6, %11, %15, %6;\n\t" /* (a.d4 * b.d1).hi */

      "mad.lo.cc.u32   %5, %12, %14, %5;\n\t" /* (a.d5 * b.d0).lo */
      "madc.lo.u32     %6, %12, %15, %6;\n\t" /* (a.d5 * b.d1).lo */

      "mad.hi.u32      %6, %12, %14, %6;\n\t" /* (a.d5 * b.d0).hi */

      "mad.lo.u32      %6, %13, %14, %6;\n\t" /* (a.d6 * b.d0).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5), "=r" (res->d6)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6),
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3), "r" (b.d4), "r" (b.d5), "r" (b.d6));
}


__device__ static void mul_224_192(int224 *res, int224 a, int192 b)
/* res = a * b (only lower 224 bits of the result) */
{
  asm("{\n\t"
      "mul.lo.u32      %0, %7, %14;\n\t"     /* (a.d0 * b.d0).lo */
      "mul.lo.u32      %1, %7, %15;\n\t"     /* (a.d0 * b.d1).lo */
      "mul.lo.u32      %2, %7, %16;\n\t"     /* (a.d0 * b.d2).lo */
      "mul.lo.u32      %3, %7, %17;\n\t"     /* (a.d0 * b.d3).lo */
      "mul.lo.u32      %4, %7, %18;\n\t"     /* (a.d0 * b.d4).lo */
      "mul.lo.u32      %5, %7, %19;\n\t"     /* (a.d0 * b.d5).lo */
      "mul.lo.u32      %6, %13,%14;\n\t"     /* (a.d6 * b.d0).lo */

      "mad.hi.cc.u32   %1, %7, %14, %1;\n\t" /* (a.d0 * b.d0).hi */
      "madc.hi.cc.u32  %2, %7, %15, %2;\n\t" /* (a.d0 * b.d1).hi */
      "madc.hi.cc.u32  %3, %7, %16, %3;\n\t" /* (a.d0 * b.d2).hi */
      "madc.hi.cc.u32  %4, %7, %17, %4;\n\t" /* (a.d0 * b.d3).hi */
      "madc.hi.cc.u32  %5, %7, %18, %5;\n\t" /* (a.d0 * b.d4).hi */
      "madc.hi.u32     %6, %7, %19, %6;\n\t" /* (a.d0 * b.d5).hi */

      "mad.lo.cc.u32   %1, %8, %14, %1;\n\t" /* (a.d1 * b.d0).lo */
      "madc.lo.cc.u32  %2, %8, %15, %2;\n\t" /* (a.d1 * b.d1).lo */
      "madc.lo.cc.u32  %3, %8, %16, %3;\n\t" /* (a.d1 * b.d2).lo */
      "madc.lo.cc.u32  %4, %8, %17, %4;\n\t" /* (a.d1 * b.d3).lo */
      "madc.lo.cc.u32  %5, %8, %18, %5;\n\t" /* (a.d1 * b.d4).lo */
      "madc.lo.u32     %6, %8, %19, %6;\n\t" /* (a.d1 * b.d5).lo */

      "mad.hi.cc.u32   %2, %8, %14, %2;\n\t" /* (a.d1 * b.d0).hi */
      "madc.hi.cc.u32  %3, %8, %15, %3;\n\t" /* (a.d1 * b.d1).hi */
      "madc.hi.cc.u32  %4, %8, %16, %4;\n\t" /* (a.d1 * b.d2).hi */
      "madc.hi.cc.u32  %5, %8, %17, %5;\n\t" /* (a.d1 * b.d3).hi */
      "madc.hi.u32     %6, %8, %18, %6;\n\t" /* (a.d1 * b.d4).hi */

      "mad.lo.cc.u32   %2, %9, %14, %2;\n\t" /* (a.d2 * b.d0).lo */
      "madc.lo.cc.u32  %3, %9, %15, %3;\n\t" /* (a.d2 * b.d1).lo */
      "madc.lo.cc.u32  %4, %9, %16, %4;\n\t" /* (a.d2 * b.d2).lo */
      "madc.lo.cc.u32  %5, %9, %17, %5;\n\t" /* (a.d2 * b.d3).lo */
      "madc.lo.u32     %6, %9, %18, %6;\n\t" /* (a.d2 * b.d4).lo */

      "mad.hi.cc.u32   %3, %9, %14, %3;\n\t" /* (a.d2 * b.d0).hi */
      "madc.hi.cc.u32  %4, %9, %15, %4;\n\t" /* (a.d2 * b.d1).hi */
      "madc.hi.cc.u32  %5, %9, %16, %5;\n\t" /* (a.d2 * b.d2).hi */
      "madc.hi.u32     %6, %9, %17, %6;\n\t" /* (a.d2 * b.d3).hi */

      "mad.lo.cc.u32   %3, %10, %14, %3;\n\t" /* (a.d3 * b.d0).lo */
      "madc.lo.cc.u32  %4, %10, %15, %4;\n\t" /* (a.d3 * b.d1).lo */
      "madc.lo.cc.u32  %5, %10, %16, %5;\n\t" /* (a.d3 * b.d2).lo */
      "madc.lo.u32     %6, %10, %17, %6;\n\t" /* (a.d3 * b.d3).lo */

      "mad.hi.cc.u32   %4, %10, %14, %4;\n\t" /* (a.d3 * b.d0).hi */
      "madc.hi.cc.u32  %5, %10, %15, %5;\n\t" /* (a.d3 * b.d1).hi */
      "madc.hi.u32     %6, %10, %16, %6;\n\t" /* (a.d3 * b.d2).hi */

      "mad.lo.cc.u32   %4, %11, %14, %4;\n\t" /* (a.d4 * b.d0).lo */
      "madc.lo.cc.u32  %5, %11, %15, %5;\n\t" /* (a.d4 * b.d1).lo */
      "madc.lo.u32     %6, %11, %16, %6;\n\t" /* (a.d4 * b.d2).lo */

      "mad.hi.cc.u32   %5, %11, %14, %5;\n\t" /* (a.d4 * b.d0).hi */
      "madc.hi.u32     %6, %11, %15, %6;\n\t" /* (a.d4 * b.d1).hi */

      "mad.lo.cc.u32   %5, %12, %14, %5;\n\t" /* (a.d5 * b.d0).lo */
      "madc.lo.u32     %6, %12, %15, %6;\n\t" /* (a.d5 * b.d1).lo */

      "mad.hi.u32      %6, %12, %14, %6;\n\t" /* (a.d5 * b.d0).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5), "=r" (res->d6)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6),
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3), "r" (b.d4), "r" (b.d5));
}


#if 0
__device__ static void mulsub_224_192(int224 *res, int416 c, int224 a, int192 negb)
/* res = c - a * b (only lower 224 bits of the result), negb.d6 is 0xFFFFFFFF */
{
  asm("{\n\t"
      "mad.lo.cc.u32   %0, %7, %14, %20;\n\t"  /* c += (a.d0 * negb.d0).lo */
      "madc.lo.cc.u32  %1, %7, %15, %21;\n\t"  /* c += (a.d0 * negb.d1).lo */
      "madc.lo.cc.u32  %2, %7, %16, %22;\n\t"  /* c += (a.d0 * negb.d2).lo */
      "madc.lo.cc.u32  %3, %7, %17, %23;\n\t"  /* c += (a.d0 * negb.d3).lo */
      "madc.lo.cc.u32  %4, %7, %18, %24;\n\t"  /* c += (a.d0 * negb.d4).lo */
      "madc.lo.cc.u32  %5, %7, %19, %25;\n\t"  /* c += (a.d0 * negb.d5).lo */
      "madc.lo.u32     %6, %13, %14, %26;\n\t" /* c += (a.d6 * negb.d0).lo */

      "mad.hi.cc.u32   %1, %7, %14, %1;\n\t"   /* c += (a.d0 * negb.d0).hi */
      "madc.hi.cc.u32  %2, %7, %15, %2;\n\t"   /* c += (a.d0 * negb.d1).hi */
      "madc.hi.cc.u32  %3, %7, %16, %3;\n\t"   /* c += (a.d0 * negb.d2).hi */
      "madc.hi.cc.u32  %4, %7, %17, %4;\n\t"   /* c += (a.d0 * negb.d3).hi */
      "madc.hi.cc.u32  %5, %7, %18, %5;\n\t"   /* c += (a.d0 * negb.d4).hi */
      "madc.hi.u32     %6, %7, %19, %6;\n\t"   /* c += (a.d0 * negb.d5).hi */

      "mad.lo.cc.u32   %1, %8, %14, %1;\n\t"   /* c += (a.d1 * negb.d0).lo */
      "madc.lo.cc.u32  %2, %8, %15, %2;\n\t"   /* c += (a.d1 * negb.d1).lo */
      "madc.lo.cc.u32  %3, %8, %16, %3;\n\t"   /* c += (a.d1 * negb.d2).lo */
      "madc.lo.cc.u32  %4, %8, %17, %4;\n\t"   /* c += (a.d1 * negb.d3).lo */
      "madc.lo.cc.u32  %5, %8, %18, %5;\n\t"   /* c += (a.d1 * negb.d4).lo */
      "madc.lo.u32     %6, %8, %19, %6;\n\t"   /* c += (a.d1 * negb.d5).lo */

      "mad.hi.cc.u32   %2, %8, %14, %2;\n\t"   /* c += (a.d1 * negb.d0).hi */
      "madc.hi.cc.u32  %3, %8, %15, %3;\n\t"   /* c += (a.d1 * negb.d1).hi */
      "madc.hi.cc.u32  %4, %8, %16, %4;\n\t"   /* c += (a.d1 * negb.d2).hi */
      "madc.hi.cc.u32  %5, %8, %17, %5;\n\t"   /* c += (a.d1 * negb.d3).hi */
      "madc.hi.u32     %6, %8, %18, %6;\n\t"   /* c += (a.d1 * negb.d4).hi */

      "mad.lo.cc.u32   %2, %9, %14, %2;\n\t"   /* c += (a.d2 * negb.d0).lo */
      "madc.lo.cc.u32  %3, %9, %15, %3;\n\t"   /* c += (a.d2 * negb.d1).lo */
      "madc.lo.cc.u32  %4, %9, %16, %4;\n\t"   /* c += (a.d2 * negb.d2).lo */
      "madc.lo.cc.u32  %5, %9, %17, %5;\n\t"   /* c += (a.d2 * negb.d3).lo */
      "madc.lo.u32     %6, %9, %18, %6;\n\t"   /* c += (a.d2 * negb.d4).lo */

      "mad.hi.cc.u32   %3, %9, %14, %3;\n\t"   /* c += (a.d2 * negb.d0).hi */
      "madc.hi.cc.u32  %4, %9, %15, %4;\n\t"   /* c += (a.d2 * negb.d1).hi */
      "madc.hi.cc.u32  %5, %9, %16, %5;\n\t"   /* c += (a.d2 * negb.d2).hi */
      "madc.hi.u32     %6, %9, %17, %6;\n\t"   /* c += (a.d2 * negb.d3).hi */

      "mad.lo.cc.u32   %3, %10, %14, %3;\n\t"   /* c += (a.d3 * negb.d0).lo */
      "madc.lo.cc.u32  %4, %10, %15, %4;\n\t"   /* c += (a.d3 * negb.d1).lo */
      "madc.lo.cc.u32  %5, %10, %16, %5;\n\t"   /* c += (a.d3 * negb.d2).lo */
      "madc.lo.u32     %6, %10, %17, %6;\n\t"   /* c += (a.d3 * negb.d3).lo */

      "mad.hi.cc.u32   %4, %10, %14, %4;\n\t"   /* c += (a.d3 * negb.d0).hi */
      "madc.hi.cc.u32  %5, %10, %15, %5;\n\t"   /* c += (a.d3 * negb.d1).hi */
      "madc.hi.u32     %6, %10, %16, %6;\n\t"   /* c += (a.d3 * negb.d2).hi */

      "mad.lo.cc.u32   %4, %11, %14, %4;\n\t"   /* c += (a.d4 * negb.d0).lo */
      "madc.lo.cc.u32  %5, %11, %15, %5;\n\t"   /* c += (a.d4 * negb.d1).lo */
      "madc.lo.u32     %6, %11, %16, %6;\n\t"   /* c += (a.d4 * negb.d2).lo */

      "mad.hi.cc.u32   %5, %11, %14, %5;\n\t"   /* c += (a.d4 * negb.d0).hi */
      "madc.hi.u32     %6, %11, %15, %6;\n\t"   /* c += (a.d4 * negb.d1).hi */

      "mad.lo.cc.u32   %5, %12, %14, %5;\n\t"  /* c += (a.d5 * negb.d0).lo */
      "madc.lo.u32     %6, %12, %15, %6;\n\t"  /* c += (a.d5 * negb.d1).lo */

      "mad.hi.u32      %6, %12, %14, %6;\n\t"  /* c += (a.d5 * negb.d0).hi */

      "sub.u32         %6, %6, %7;\n\t"        /* c += (a.d0 * negb.d6).lo (d6=-1) */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5), "=r" (res->d6)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4), "r" (a.d5), "r" (a.d6),
        "r" (negb.d0), "r" (negb.d1), "r" (negb.d2), "r" (negb.d3), "r" (negb.d4), "r" (negb.d5),
        "r" (c.d0), "r" (c.d1), "r" (c.d2), "r" (c.d3), "r" (c.d4), "r" (c.d5), "r" (c.d6));
}
#endif


__device__ static void div_480_224(int256 *res, int480 q, int224 n, float nf)
/* res = q / n (integer division) */
{
  float qf;
  unsigned int qi;
  int480 nn;
  int224 tmp224;

/********** Step X, Offset 2^235 (7*32 + 11) **********/
  qf= __uint2float_rn(q.d14);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d13);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);

  res->d7 = (qi <<= 11);

// nn = n * qi
  nn.d7 =                                 __umul32(n.d0, qi);
  nn.d8 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d10= __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d11= __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d12= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d13= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d14= __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d7 = __sub_cc (q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc_cc(q.d12, nn.d12);
  q.d13= __subc_cc(q.d13, nn.d13);
  q.d14= __subc   (q.d14, nn.d14);

/********** Step Y, Offset 2^215 (6*32 + 23) **********/
  qf= __uint2float_rn(q.d14);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qY: %X\n", qi);

  res->d6 =  qi << 23;
  res->d7 += qi >>  9;

// nn = n * qi
  nn.d6 =                                 __umul32(n.d0, qi);
  nn.d7 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d10= __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d11= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d12= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d13= __addc_cc(__umul32hi(n.d6, qi),                  0);

// shiftleft nn 23 bits
  nn.d13 = (nn.d13 << 23) + (nn.d12 >> 9);
  nn.d12 = (nn.d12 << 23) + (nn.d11 >> 9);
  nn.d11 = (nn.d11 << 23) + (nn.d10 >> 9);
  nn.d10 = (nn.d10 << 23) + (nn.d9  >> 9);
  nn.d9  = (nn.d9  << 23) + (nn.d8  >> 9);
  nn.d8  = (nn.d8  << 23) + (nn.d7  >> 9);
  nn.d7  = (nn.d7  << 23) + (nn.d6  >> 9);
  nn.d6  =  nn.d6  << 23;

// q = q - nn
  q.d6 = __sub_cc (q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc_cc(q.d12, nn.d12);
  q.d13= __subc   (q.d13, nn.d13);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qYq: %X %X\n", q.d13, q.d12);

/********** Step Z, Offset 2^195 (6*32 + 3) **********/
  qf= __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qZ: %X\n", qi);

  qi <<= 3;
  res->d6 = __add_cc(res->d6, qi);
  res->d7 = __addc  (res->d7, 0);

// nn = n * qi
  nn.d6 =                                 __umul32(n.d0, qi);
  nn.d7 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d10= __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d11= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d12= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d13= __addc_cc(__umul32hi(n.d6, qi),                  0);

// q = q - nn
  q.d6 = __sub_cc (q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc_cc(q.d12, nn.d12);
  q.d13= __subc   (q.d13, nn.d13);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qZq: %X %X\n", q.d13, q.d12);

/********** Step 1, Offset 2^175 (5*32 + 15) **********/
  qf= __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d11);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) if ((qi >> 17) != 0) printf ("1/f fail 1\n");

  res->d5 = qi << 15;
  res->d6 = __add_cc(res->d6, qi >> 17);
  res->d7 = __addc  (res->d7, 0);

// nn = n * qi
  nn.d5 =                                 __umul32(n.d0, qi);
  nn.d6 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d10= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d11= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d12= __addc_cc(__umul32hi(n.d6, qi),                  0);

//if (nn.d11 >> 17 != q.d12) printf ("1/f fail 1a\n");
// shiftleft nn 15 bits
//nn.d13=                 (nn.d12>> 17);
  nn.d12= (nn.d12<< 15) + (nn.d11>> 17);
  nn.d11= (nn.d11<< 15) + (nn.d10>> 17);
  nn.d10= (nn.d10<< 15) + (nn.d9 >> 17);
  nn.d9 = (nn.d9 << 15) + (nn.d8 >> 17);
  nn.d8 = (nn.d8 << 15) + (nn.d7 >> 17);
  nn.d7 = (nn.d7 << 15) + (nn.d6 >> 17);
  nn.d6 = (nn.d6 << 15) + (nn.d5 >> 17);
  nn.d5 =  nn.d5 << 15;
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("q1nn: %X %X %X\n", nn.d12, nn.d11, nn.d10);

// q = q - nn
  q.d5 = __sub_cc (q.d5 , nn.d5 );
  q.d6 = __subc_cc(q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc   (q.d12, nn.d12);

/********** Step 2, Offset 2^155 (4*32 + 27) **********/
  qf= __uint2float_rn(q.d12);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf*= 32.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("q2: %X\n", qi);

  res->d4 = qi<<27;
  res->d5 = __add_cc(res->d5, qi>>5);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d10 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d11 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d10 >> 5 != q.d11) printf ("1/f fail 2\n");
// shiftleft nn 27 bits
  nn.d11 = (nn.d11 << 27) + (nn.d10 >> 5);
  nn.d10 = (nn.d10 << 27) + (nn.d9  >> 5);
  nn.d9  = (nn.d9  << 27) + (nn.d8  >> 5);
  nn.d8  = (nn.d8  << 27) + (nn.d7  >> 5);
  nn.d7  = (nn.d7  << 27) + (nn.d6  >> 5);
  nn.d6  = (nn.d6  << 27) + (nn.d5  >> 5);
  nn.d5  = (nn.d5  << 27) + (nn.d4  >> 5);
  nn.d4  =  nn.d4  << 27;

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc_cc(q.d10, nn.d10);
  q.d11 = __subc   (q.d11, nn.d11);

/********** Step 3, Offset 2^135 (4*32 + 7) **********/
  qf= __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf*= 33554432.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q3: %X\n", qi);

  qi <<= 7;
  res->d4 = __add_cc (res->d4, qi);
  res->d5 = __addc_cc(res->d5,  0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d10 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d11 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc_cc(q.d10, nn.d10);
  q.d11 = __subc   (q.d11, nn.d11);

/********** Step 4, Offset 2^115 (3*32 + 19) **********/
  qf= __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf*= 8192.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q4: %X\n", qi);

  res->d3 = qi << 19;
  res->d4 = __add_cc (res->d4, qi >> 13);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d3 =                                 __umul32(n.d0, qi);
  nn.d4 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d10= __addc   (__umul32hi(n.d6, qi),                  0);

// shiftleft nn 19 bits
  nn.d10 = (nn.d10 << 19)+ (nn.d9 >> 13);
  nn.d9  = (nn.d9 << 19) + (nn.d8 >> 13);
  nn.d8  = (nn.d8 << 19) + (nn.d7 >> 13);
  nn.d7  = (nn.d7 << 19) + (nn.d6 >> 13);
  nn.d6  = (nn.d6 << 19) + (nn.d5 >> 13);
  nn.d5  = (nn.d5 << 19) + (nn.d4 >> 13);
  nn.d4  = (nn.d4 << 19) + (nn.d3 >> 13);
  nn.d3  =  nn.d3 << 19;

//  q = q - nn
  q.d3  = __sub_cc (q.d3,  nn.d3);
  q.d4  = __subc_cc(q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc   (q.d10, nn.d10);
//if (0 != q.d11) printf ("1/f fail 4d\n");

/********** Step 5, Offset 2^95 (2*32 + 31) **********/
  qf= __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 2.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q5: %X\n", qi);

  res->d2 = qi << 31;
  res->d3 = __add_cc (res->d3, qi >> 1);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d9 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (nn.d9 >> 1 != q.d9) printf ("1/f fail 7\n");
// shiftleft nn 31 bits
  nn.d9 = (nn.d9 << 31) + (nn.d8 >> 1);
  nn.d8 = (nn.d8 << 31) + (nn.d7 >> 1);
  nn.d7 = (nn.d7 << 31) + (nn.d6 >> 1);
  nn.d6 = (nn.d6 << 31) + (nn.d5 >> 1);
  nn.d5 = (nn.d5 << 31) + (nn.d4 >> 1);
  nn.d4 = (nn.d4 << 31) + (nn.d3 >> 1);
  nn.d3 = (nn.d3 << 31) + (nn.d2 >> 1);
  nn.d2 =  nn.d2 << 31;

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc_cc(q.d8, nn.d8);
  q.d9 = __subc   (q.d9, nn.d9);

/********** Step 6, Offset 2^75 (2*32 + 11) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q6: %X\n", qi);

  qi <<= 11;
  res->d2 = __add_cc (res->d2, qi);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d9 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc_cc(q.d8, nn.d8);
  q.d9 = __subc   (q.d9, nn.d9);

/********** Step 7, Offset 2^55 (1*32 + 23) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q7: %X\n", qi);

  res->d1 = qi << 23;
  res->d2 = __add_cc (res->d2, qi >> 9);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d8 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (nn.d8 >> 9 != q.d9) printf ("1/f fail 7b\n");
// shiftleft nn 23 bits
//nn.d9 =                  nn.d8 >> 9;
  nn.d8 = (nn.d8 << 23) + (nn.d7 >> 9);
  nn.d7 = (nn.d7 << 23) + (nn.d6 >> 9);
  nn.d6 = (nn.d6 << 23) + (nn.d5 >> 9);
  nn.d5 = (nn.d5 << 23) + (nn.d4 >> 9);
  nn.d4 = (nn.d4 << 23) + (nn.d3 >> 9);
  nn.d3 = (nn.d3 << 23) + (nn.d2 >> 9);
  nn.d2 = (nn.d2 << 23) + (nn.d1 >> 9);
  nn.d1 =  nn.d1 << 23;

// q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc   (q.d8, nn.d8);
//q.d8 = __subc_cc(q.d8, nn.d8);
//q.d9 = __subc   (q.d9, nn.d9);

/********** Step 8, Offset 2^35 (1*32 + 3) **********/

  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q8: %X\n", qi);

  qi <<= 3;
  res->d1 = __add_cc (res->d1, qi);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d8 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc   (q.d8, nn.d8);

/********** Step 9, Offset 2^15 (0*32 + 15) **********/

  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q9: %X\n", qi);

  res->d0 = qi << 15;
  res->d1 = __add_cc (res->d1, qi >> 17);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d0 =                                 __umul32(n.d0, qi);
  nn.d1 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d7 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d6 >> 17 != q.d7) printf ("1/f fail 9\n");

// shiftleft nn 15 bits
//nn.d8 =                  nn.d7 >> 17;
  nn.d7 = (nn.d7 << 15) + (nn.d6 >> 17);
  nn.d6 = (nn.d6 << 15) + (nn.d5 >> 17);
  nn.d5 = (nn.d5 << 15) + (nn.d4 >> 17);
  nn.d4 = (nn.d4 << 15) + (nn.d3 >> 17);
  nn.d3 = (nn.d3 << 15) + (nn.d2 >> 17);
  nn.d2 = (nn.d2 << 15) + (nn.d1 >> 17);
  nn.d1 = (nn.d1 << 15) + (nn.d0 >> 17);
  nn.d0 =  nn.d0 << 15;

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 10, Offset 2^0 (0*32 + 0) **********/

  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q10: %X\n", qi);

  res->d0 = __add_cc (res->d0, qi);
  res->d1 = __addc_cc(res->d1, 0);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc_cc(res->d6, 0);
  res->d7 = __addc   (res->d7, 0);

// nn = n * qi
  nn.d0 =                                  __umul32(n.d0, qi);
  nn.d1 = __add_cc  (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc (__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc (__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc (__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc_cc (__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d6 = __addc    (__umul32hi(n.d5, qi), __umul32(n.d6, qi));

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/*
qi is always a little bit too small, this is OK for all steps except the last
one. Sometimes the result is a little bit bigger than n
*/

//if (blockIdx.x == 12 && threadIdx.x == 4)
//printf ("  rem: %X %X %X %X %X %X %X\r\n", q.d6, q.d5, q.d4, q.d3, q.d2, q.d1, q.d0);

  tmp224.d0=q.d0;
  tmp224.d1=q.d1;
  tmp224.d2=q.d2;
  tmp224.d3=q.d3;
  tmp224.d4=q.d4;
  tmp224.d5=q.d5;
  tmp224.d6=q.d6;

  if(cmp_ge_224(tmp224,n))
  {
    res->d0 = __add_cc (res->d0, 1);
    res->d1 = __addc_cc(res->d1, 0);
    res->d2 = __addc_cc(res->d2, 0);
    res->d3 = __addc_cc(res->d3, 0);
    res->d4 = __addc_cc(res->d4, 0);
    res->d5 = __addc_cc(res->d5, 0);
    res->d6 = __addc_cc(res->d6, 0);
    res->d7 = __addc   (res->d7, 0);
  }
}


__device__ static void div_448_224(int224 *res, int448 q, int224 n, float nf)
/* res = q / n (integer division) */
{
  float qf;
  unsigned int qi;
  int448 nn;
  int224 tmp224;

/********** Step Y, Offset 2^215 (6*32 + 23) **********/
  qf= __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qY: %X\n", qi);

  res->d6 = qi << 23;

// nn = n * qi
  nn.d6 =                                 __umul32(n.d0, qi);
  nn.d7 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d10= __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d11= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d12= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d13= __addc_cc(__umul32hi(n.d6, qi),                  0);

// shiftleft nn 23 bits
  nn.d13 = (nn.d13 << 23) + (nn.d12 >> 9);
  nn.d12 = (nn.d12 << 23) + (nn.d11 >> 9);
  nn.d11 = (nn.d11 << 23) + (nn.d10 >> 9);
  nn.d10 = (nn.d10 << 23) + (nn.d9  >> 9);
  nn.d9  = (nn.d9  << 23) + (nn.d8  >> 9);
  nn.d8  = (nn.d8  << 23) + (nn.d7  >> 9);
  nn.d7  = (nn.d7  << 23) + (nn.d6  >> 9);
  nn.d6  =  nn.d6  << 23;

// q = q - nn
  q.d6 = __sub_cc (q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc_cc(q.d12, nn.d12);
  q.d13= __subc   (q.d13, nn.d13);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qYq: %X %X\n", q.d13, q.d12);

/********** Step Z, Offset 2^195 (6*32 + 3) **********/
  qf= __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qZ: %X\n", qi);

  qi <<= 3;
  res->d6 += qi;

// nn = n * qi
  nn.d6 =                                 __umul32(n.d0, qi);
  nn.d7 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d10= __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d11= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d12= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d13= __addc_cc(__umul32hi(n.d6, qi),                  0);

// q = q - nn
  q.d6 = __sub_cc (q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc_cc(q.d12, nn.d12);
  q.d13= __subc   (q.d13, nn.d13);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("qZq: %X %X\n", q.d13, q.d12);

/********** Step 1, Offset 2^175 (5*32 + 15) **********/
  qf= __uint2float_rn(q.d13);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d12);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d11);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) if ((qi >> 17) != 0) printf ("1/f fail 1\n");
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q1: %X\n", qi);
  res->d5 = qi << 15;
  res->d6 += qi >> 17;

// nn = n * qi
  nn.d5 =                                 __umul32(n.d0, qi);
  nn.d6 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d10= __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d11= __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d12= __addc_cc(__umul32hi(n.d6, qi),                  0);

//if (nn.d11 >> 17 != q.d12) printf ("1/f fail 1a\n");
// shiftleft nn 15 bits
//nn.d13=                 (nn.d12>> 17);
  nn.d12= (nn.d12<< 15) + (nn.d11>> 17);
  nn.d11= (nn.d11<< 15) + (nn.d10>> 17);
  nn.d10= (nn.d10<< 15) + (nn.d9 >> 17);
  nn.d9 = (nn.d9 << 15) + (nn.d8 >> 17);
  nn.d8 = (nn.d8 << 15) + (nn.d7 >> 17);
  nn.d7 = (nn.d7 << 15) + (nn.d6 >> 17);
  nn.d6 = (nn.d6 << 15) + (nn.d5 >> 17);
  nn.d5 =  nn.d5 << 15;
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("q1nn: %X %X %X\n", nn.d12, nn.d11, nn.d10);

// q = q - nn
  q.d5 = __sub_cc (q.d5 , nn.d5 );
  q.d6 = __subc_cc(q.d6 , nn.d6 );
  q.d7 = __subc_cc(q.d7 , nn.d7 );
  q.d8 = __subc_cc(q.d8 , nn.d8 );
  q.d9 = __subc_cc(q.d9 , nn.d9 );
  q.d10= __subc_cc(q.d10, nn.d10);
  q.d11= __subc_cc(q.d11, nn.d11);
  q.d12= __subc   (q.d12, nn.d12);

/********** Step 2, Offset 2^155 (4*32 + 27) **********/
  qf= __uint2float_rn(q.d12);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf*= 32.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 0) printf ("q2: %X\n", qi);

  res->d4 = qi<<27;
  res->d5 = __add_cc(res->d5, qi>>5);
  res->d6 = __addc  (res->d6,  0);

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d10 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d11 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d10 >> 5 != q.d11) printf ("1/f fail 2\n");
// shiftleft nn 27 bits
  nn.d11 = (nn.d11 << 27) + (nn.d10 >> 5);
  nn.d10 = (nn.d10 << 27) + (nn.d9  >> 5);
  nn.d9  = (nn.d9  << 27) + (nn.d8  >> 5);
  nn.d8  = (nn.d8  << 27) + (nn.d7  >> 5);
  nn.d7  = (nn.d7  << 27) + (nn.d6  >> 5);
  nn.d6  = (nn.d6  << 27) + (nn.d5  >> 5);
  nn.d5  = (nn.d5  << 27) + (nn.d4  >> 5);
  nn.d4  =  nn.d4  << 27;

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc_cc(q.d10, nn.d10);
  q.d11 = __subc   (q.d11, nn.d11);

/********** Step 3, Offset 2^135 (4*32 + 7) **********/
  qf= __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf*= 33554432.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q3: %X\n", qi);

  qi <<= 7;
  res->d4 = __add_cc (res->d4, qi);
  res->d5 = __addc_cc(res->d5,  0);
  res->d6 = __addc   (res->d6,  0);

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d10 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d11 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc_cc(q.d10, nn.d10);
  q.d11 = __subc   (q.d11, nn.d11);

/********** Step 4, Offset 2^115 (3*32 + 19) **********/
  qf= __uint2float_rn(q.d11);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf*= 8192.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q4: %X\n", qi);

  res->d3 = qi << 19;
  res->d4 = __add_cc (res->d4, qi >> 13);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d3 =                                 __umul32(n.d0, qi);
  nn.d4 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d9 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d10= __addc   (__umul32hi(n.d6, qi),                  0);

// shiftleft nn 19 bits
  nn.d10 = (nn.d10 << 19)+ (nn.d9 >> 13);
  nn.d9  = (nn.d9 << 19) + (nn.d8 >> 13);
  nn.d8  = (nn.d8 << 19) + (nn.d7 >> 13);
  nn.d7  = (nn.d7 << 19) + (nn.d6 >> 13);
  nn.d6  = (nn.d6 << 19) + (nn.d5 >> 13);
  nn.d5  = (nn.d5 << 19) + (nn.d4 >> 13);
  nn.d4  = (nn.d4 << 19) + (nn.d3 >> 13);
  nn.d3  =  nn.d3 << 19;

//  q = q - nn
  q.d3  = __sub_cc (q.d3,  nn.d3);
  q.d4  = __subc_cc(q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc_cc(q.d9,  nn.d9);
  q.d10 = __subc   (q.d10, nn.d10);
//if (0 != q.d11) printf ("1/f fail 4d\n");

/********** Step 5, Offset 2^95 (2*32 + 31) **********/
  qf= __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 2.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q5: %X\n", qi);

  res->d2 = qi << 31;
  res->d3 = __add_cc (res->d3, qi >> 1);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d9 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (nn.d9 >> 1 != q.d9) printf ("1/f fail 7\n");
// shiftleft nn 31 bits
  nn.d9 = (nn.d9 << 31) + (nn.d8 >> 1);
  nn.d8 = (nn.d8 << 31) + (nn.d7 >> 1);
  nn.d7 = (nn.d7 << 31) + (nn.d6 >> 1);
  nn.d6 = (nn.d6 << 31) + (nn.d5 >> 1);
  nn.d5 = (nn.d5 << 31) + (nn.d4 >> 1);
  nn.d4 = (nn.d4 << 31) + (nn.d3 >> 1);
  nn.d3 = (nn.d3 << 31) + (nn.d2 >> 1);
  nn.d2 =  nn.d2 << 31;

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc_cc(q.d8, nn.d8);
  q.d9 = __subc   (q.d9, nn.d9);

/********** Step 6, Offset 2^75 (2*32 + 11) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q6: %X\n", qi);

  qi <<= 11;
  res->d2 = __add_cc (res->d2, qi);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d9 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc_cc(q.d8, nn.d8);
  q.d9 = __subc   (q.d9, nn.d9);

/********** Step 7, Offset 2^55 (1*32 + 23) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q7: %X\n", qi);

  res->d1 = qi << 23;
  res->d2 = __add_cc (res->d2, qi >> 9);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d8 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (nn.d8 >> 9 != q.d9) printf ("1/f fail 7b\n");
// shiftleft nn 23 bits
  nn.d9 =                  nn.d8 >> 9;
  nn.d8 = (nn.d8 << 23) + (nn.d7 >> 9);
  nn.d7 = (nn.d7 << 23) + (nn.d6 >> 9);
  nn.d6 = (nn.d6 << 23) + (nn.d5 >> 9);
  nn.d5 = (nn.d5 << 23) + (nn.d4 >> 9);
  nn.d4 = (nn.d4 << 23) + (nn.d3 >> 9);
  nn.d3 = (nn.d3 << 23) + (nn.d2 >> 9);
  nn.d2 = (nn.d2 << 23) + (nn.d1 >> 9);
  nn.d1 =  nn.d1 << 23;

// q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc   (q.d8, nn.d8);
//q.d8 = __subc_cc(q.d8, nn.d8);
//q.d9 = __subc   (q.d9, nn.d9);

/********** Step 8, Offset 2^35 (1*32 + 3) **********/

  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q8: %X\n", qi);

  qi <<= 3;
  res->d1 = __add_cc (res->d1, qi);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d8 = __addc   (__umul32hi(n.d6, qi),                  0);

//  q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc_cc(q.d7, nn.d7);
  q.d8 = __subc   (q.d8, nn.d8);

/********** Step 9, Offset 2^15 (0*32 + 15) **********/

  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q9: %X\n", qi);

  res->d0 = qi << 15;
  res->d1 = __add_cc (res->d1, qi >> 17);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d0 =                                 __umul32(n.d0, qi);
  nn.d1 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d5, qi), __umul32(n.d6, qi));
  nn.d7 = __addc   (__umul32hi(n.d6, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d6 >> 17 != q.d7) printf ("1/f fail 9\n");

// shiftleft nn 15 bits
//nn.d8 =                  nn.d7 >> 17;
  nn.d7 = (nn.d7 << 15) + (nn.d6 >> 17);
  nn.d6 = (nn.d6 << 15) + (nn.d5 >> 17);
  nn.d5 = (nn.d5 << 15) + (nn.d4 >> 17);
  nn.d4 = (nn.d4 << 15) + (nn.d3 >> 17);
  nn.d3 = (nn.d3 << 15) + (nn.d2 >> 17);
  nn.d2 = (nn.d2 << 15) + (nn.d1 >> 17);
  nn.d1 = (nn.d1 << 15) + (nn.d0 >> 17);
  nn.d0 =  nn.d0 << 15;

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 10, Offset 2^0 (0*32 + 0) **********/

  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q10: %X\n", qi);

  res->d0 = __add_cc (res->d0, qi);
  res->d1 = __addc_cc(res->d1, 0);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc_cc(res->d5, 0);
  res->d6 = __addc   (res->d6, 0);

// nn = n * qi
  nn.d0 =                                  __umul32(n.d0, qi);
  nn.d1 = __add_cc  (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc (__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc (__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc (__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc_cc (__umul32hi(n.d4, qi), __umul32(n.d5, qi));
  nn.d6 = __addc    (__umul32hi(n.d5, qi), __umul32(n.d6, qi));

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/*
qi is always a little bit too small, this is OK for all steps except the last
one. Sometimes the result is a little bit bigger than n
*/

//if (blockIdx.x == 12 && threadIdx.x == 4)
//printf ("  rem: %X %X %X %X %X %X %X\r\n", q.d6, q.d5, q.d4, q.d3, q.d2, q.d1, q.d0);

  tmp224.d0=q.d0;
  tmp224.d1=q.d1;
  tmp224.d2=q.d2;
  tmp224.d3=q.d3;
  tmp224.d4=q.d4;
  tmp224.d5=q.d5;
  tmp224.d6=q.d6;

  if(cmp_ge_224(tmp224,n))
  {
    res->d0 = __add_cc (res->d0, 1);
    res->d1 = __addc_cc(res->d1, 0);
    res->d2 = __addc_cc(res->d2, 0);
    res->d3 = __addc_cc(res->d3, 0);
    res->d4 = __addc_cc(res->d4, 0);
    res->d5 = __addc_cc(res->d5, 0);
    res->d6 = __addc   (res->d6, 0);
  }
}


__device__ static void div_416_224(int224 *res, int416 q, int224 n, float nf)
/* res = q / n (integer division) */
{
	int448 tmp448;

	tmp448.d0 = q.d0;
	tmp448.d1 = q.d1;
	tmp448.d2 = q.d2;
	tmp448.d3 = q.d3;
	tmp448.d4 = q.d4;
	tmp448.d5 = q.d5;
	tmp448.d6 = q.d6;
	tmp448.d7 = q.d7;
	tmp448.d8 = q.d8;
	tmp448.d9 = q.d9;
	tmp448.d10 = q.d10;
	tmp448.d11 = q.d11;
	tmp448.d12 = q.d12;
	tmp448.d13 = 0;
	div_448_224(res, tmp448, n, nf);
}


__device__ static void mod_simple_224_192(int192 *res, int224 q, int192 n, float nf)
/*
res = q mod n
used for refinement in barrett modular multiplication
assumes q < Xn where X is a small integer
*/
{
  float qf;
  unsigned int qi;
  int224 nn;

  qf = __uint2float_rn(q.d6);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d4);

  qi=__float2uint_rz(qf*nf);

  nn.d0 =                           __umul32(n.d0, qi);
  nn.d1 = __umad32hi_cc  (n.d0, qi, __umul32(n.d1, qi));
  nn.d2 = __umad32hic_cc (n.d1, qi, __umul32(n.d2, qi));
  nn.d3 = __umad32hic_cc (n.d2, qi, __umul32(n.d3, qi));
  nn.d4 = __umad32hic_cc (n.d3, qi, __umul32(n.d4, qi));
  nn.d5 = __umad32hic_cc (n.d4, qi, __umul32(n.d5, qi));
  nn.d6 = __umad32hic    (n.d5, qi,                  0);

  res->d0 = __sub_cc (q.d0, nn.d0);
  res->d1 = __subc_cc(q.d1, nn.d1);
  res->d2 = __subc_cc(q.d2, nn.d2);
  res->d3 = __subc_cc(q.d3, nn.d3);
  res->d4 = __subc_cc(q.d4, nn.d4);
  res->d5 = __subc_cc(q.d5, nn.d5);
  q.d6    = __subc   (q.d6, nn.d6);

  if(q.d6 || cmp_ge_192(*res, n))		// final adjustment in case finalrem >= f
  {
    sub_192(res, *res, n);
  }
}


__device__ static void mod_simple_224(int224 *res, int224 q, int224 n, float nf)
/*
res = q mod n
used for refinement in barrett modular multiplication
assumes q < Xn where X is a small integer
*/
{
  float qf;
  unsigned int qi;
  int224 nn;

  qf = __uint2float_rn(q.d6);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d5);

  qi=__float2uint_rz(qf*nf);

  nn.d0 =                           __umul32(n.d0, qi);
  nn.d1 = __umad32hi_cc  (n.d0, qi, __umul32(n.d1, qi));
  nn.d2 = __umad32hic_cc (n.d1, qi, __umul32(n.d2, qi));
  nn.d3 = __umad32hic_cc (n.d2, qi, __umul32(n.d3, qi));
  nn.d4 = __umad32hic_cc (n.d3, qi, __umul32(n.d4, qi));
  nn.d5 = __umad32hic_cc (n.d4, qi, __umul32(n.d5, qi));
  nn.d6 = __umad32hic    (n.d5, qi, __umul32(n.d6, qi));

  sub_224(res, q, nn);

  if(cmp_ge_224(*res, n))			// final adjustment in case finalrem >= f
  {
    sub_224(res, *res, n);
  }
}
