/*
This file is part of mfaktc.
Copyright (C) 2012  George Woltman (woltman@alum.mit.edu)

mfaktc is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mfaktc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with mfaktc.  If not, see <http://www.gnu.org/licenses/>.
*/


__device__ static int cmp_ge_160(int160 a, int160 b)
/* checks if a is greater or equal than b */
{
  if(a.d4 != b.d4) return(a.d4 > b.d4);
  if(a.d3 != b.d3) return(a.d3 > b.d3);
  if(a.d2 != b.d2) return(a.d2 > b.d2);
  if(a.d1 != b.d1) return(a.d1 > b.d1);
  return(a.d0 >= b.d0);
}


__device__ static void sub_160(int160 *res, int160 a, int160 b)
/* a must be greater or equal b!
res = a - b */
{
  res->d0 = __sub_cc (a.d0, b.d0);
  res->d1 = __subc_cc(a.d1, b.d1);
  res->d2 = __subc_cc(a.d2, b.d2);
  res->d3 = __subc_cc(a.d3, b.d3);
  res->d4 = __subc   (a.d4, b.d4);
}


__device__ static void square_160_320(int320 *res, int160 a)
/* res = a^2, assuming that a is < 2^159 (a.d4 < 2^31)! */
{
  asm("{\n\t"
      "mul.lo.u32      %1, %10, %11;\n\t"     /* (a.d0 * a.d1).lo */
      "mul.lo.u32      %2, %10, %12;\n\t"     /* (a.d0 * a.d2).lo */
      "mul.lo.u32      %3, %10, %13;\n\t"     /* (a.d0 * a.d3).lo */
      "mul.lo.u32      %4, %10, %14;\n\t"     /* (a.d0 * a.d4).lo */

      "mad.hi.cc.u32   %2, %10, %11, %2;\n\t" /* (a.d0 * a.d1).hi */
      "madc.hi.cc.u32  %3, %10, %12, %3;\n\t" /* (a.d0 * a.d2).hi */
      "madc.hi.cc.u32  %4, %10, %13, %4;\n\t" /* (a.d0 * a.d3).hi */
      "madc.hi.u32     %5, %10, %14, 0;\n\t"  /* (a.d0 * a.d4).hi */

      "mad.lo.cc.u32   %3, %11, %12, %3;\n\t" /* (a.d1 * a.d2).lo */
      "madc.hi.cc.u32  %4, %11, %12, %4;\n\t" /* (a.d1 * a.d2).hi */
      "madc.hi.cc.u32  %5, %11, %13, %5;\n\t" /* (a.d1 * a.d3).hi */
      "madc.hi.u32     %6, %11, %14, 0;\n\t"  /* (a.d1 * a.d4).hi */

      "mad.lo.cc.u32   %4, %11, %13, %4;\n\t" /* (a.d1 * a.d3).lo */
      "madc.lo.cc.u32  %5, %11, %14, %5;\n\t" /* (a.d1 * a.d4).lo */
      "madc.hi.cc.u32  %6, %12, %13, %6;\n\t" /* (a.d2 * a.d3).hi */
      "madc.hi.u32     %7, %12, %14, 0;\n\t"  /* (a.d2 * a.d4).hi */

      "mad.lo.cc.u32   %5, %12, %13, %5;\n\t" /* (a.d2 * a.d3).lo */
      "madc.lo.cc.u32  %6, %12, %14, %6;\n\t" /* (a.d2 * a.d4).lo */
      "madc.lo.cc.u32  %7, %13, %14, %7;\n\t" /* (a.d3 * a.d4).lo */
      "madc.hi.u32     %8, %13, %14, 0;\n\t"  /* (a.d3 * a.d4).hi */

      "add.cc.u32      %1, %1, %1;\n\t"       /* Double the partial results */
      "addc.cc.u32     %2, %2, %2;\n\t"
      "addc.cc.u32     %3, %3, %3;\n\t"
      "addc.cc.u32     %4, %4, %4;\n\t"
      "addc.cc.u32     %5, %5, %5;\n\t"
      "addc.cc.u32     %6, %6, %6;\n\t"
      "addc.cc.u32     %7, %7, %7;\n\t"
      "addc.u32        %8, %8, %8;\n\t"

      "mul.lo.u32      %0, %10, %10;\n\t"       /* (a.d0 * a.d0).lo */
      "mad.hi.cc.u32   %1, %10, %10, %1;\n\t"   /* (a.d0 * a.d0).hi */
      "madc.lo.cc.u32  %2, %11, %11, %2;\n\t"   /* (a.d1 * a.d1).lo */
      "madc.hi.cc.u32  %3, %11, %11, %3;\n\t"   /* (a.d1 * a.d1).hi */
      "madc.lo.cc.u32  %4, %12, %12, %4;\n\t"   /* (a.d2 * a.d2).lo */
      "madc.hi.cc.u32  %5, %12, %12, %5;\n\t"   /* (a.d2 * a.d2).hi */
      "madc.lo.cc.u32  %6, %13, %13, %6;\n\t"   /* (a.d3 * a.d3).lo */
      "madc.hi.cc.u32  %7, %13, %13, %7;\n\t"   /* (a.d3 * a.d3).hi */
      "madc.lo.cc.u32  %8, %14, %14, %8;\n\t"   /* (a.d4 * a.d4).lo */
      "madc.hi.u32     %9, %14, %14, 0;\n\t"   /* (a.d4 * a.d4).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5),
        "=r" (res->d6), "=r" (res->d7), "=r" (res->d8), "=r" (res->d9)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4));
}


__device__ static void square_160_288(int288 *res, int160 a)
/* res = a^2, assuming that a is < 2^144 (a.d4 < 2^16)! */
{
  asm("{\n\t"
      "mul.lo.u32      %1, %9, %10;\n\t"      /* (a.d0 * a.d1).lo */
      "mul.lo.u32      %2, %9, %11;\n\t"      /* (a.d0 * a.d2).lo */
      "mul.lo.u32      %3, %9, %12;\n\t"      /* (a.d0 * a.d3).lo */
      "mul.lo.u32      %4, %9, %13;\n\t"      /* (a.d0 * a.d4).lo */

      "mad.hi.cc.u32   %2, %9, %10, %2;\n\t"  /* (a.d0 * a.d1).hi */
      "madc.hi.cc.u32  %3, %9, %11, %3;\n\t"  /* (a.d0 * a.d2).hi */
      "madc.hi.cc.u32  %4, %9, %12, %4;\n\t"  /* (a.d0 * a.d3).hi */
      "madc.hi.u32     %5, %9, %13, 0;\n\t"   /* (a.d0 * a.d4).hi */

      "mad.lo.cc.u32   %3, %10, %11, %3;\n\t" /* (a.d1 * a.d2).lo */
      "madc.hi.cc.u32  %4, %10, %11, %4;\n\t" /* (a.d1 * a.d2).hi */
      "madc.hi.cc.u32  %5, %10, %12, %5;\n\t" /* (a.d1 * a.d3).hi */
      "madc.hi.u32     %6, %10, %13, 0;\n\t"  /* (a.d1 * a.d4).hi */

      "mad.lo.cc.u32   %4, %10, %12, %4;\n\t" /* (a.d1 * a.d3).lo */
      "madc.lo.cc.u32  %5, %10, %13, %5;\n\t" /* (a.d1 * a.d4).lo */
      "madc.hi.cc.u32  %6, %11, %12, %6;\n\t" /* (a.d2 * a.d3).hi */
      "madc.hi.u32     %7, %11, %13, 0;\n\t"  /* (a.d2 * a.d4).hi */

      "mad.lo.cc.u32   %5, %11, %12, %5;\n\t" /* (a.d2 * a.d3).lo */
      "madc.lo.cc.u32  %6, %11, %13, %6;\n\t" /* (a.d2 * a.d4).lo */
      "madc.lo.cc.u32  %7, %12, %13, %7;\n\t" /* (a.d3 * a.d4).lo */
      "madc.hi.u32     %8, %12, %13, 0;\n\t"  /* (a.d3 * a.d4).hi */

      "add.cc.u32      %1, %1, %1;\n\t"       /* Double the partial results */
      "addc.cc.u32     %2, %2, %2;\n\t"
      "addc.cc.u32     %3, %3, %3;\n\t"
      "addc.cc.u32     %4, %4, %4;\n\t"
      "addc.cc.u32     %5, %5, %5;\n\t"
      "addc.cc.u32     %6, %6, %6;\n\t"
      "addc.cc.u32     %7, %7, %7;\n\t"
      "addc.u32        %8, %8, %8;\n\t"

      "mul.lo.u32      %0, %9, %9;\n\t"       /* (a.d0 * a.d0).lo */
      "mad.hi.cc.u32   %1, %9, %9, %1;\n\t"   /* (a.d0 * a.d0).hi */
      "madc.lo.cc.u32  %2, %10, %10, %2;\n\t" /* (a.d1 * a.d1).lo */
      "madc.hi.cc.u32  %3, %10, %10, %3;\n\t" /* (a.d1 * a.d1).hi */
      "madc.lo.cc.u32  %4, %11, %11, %4;\n\t" /* (a.d2 * a.d2).lo */
      "madc.hi.cc.u32  %5, %11, %11, %5;\n\t" /* (a.d2 * a.d2).hi */
      "madc.lo.cc.u32  %6, %12, %12, %6;\n\t" /* (a.d3 * a.d3).lo */
      "madc.hi.cc.u32  %7, %12, %12, %7;\n\t" /* (a.d3 * a.d3).hi */
      "madc.lo.u32     %8, %13, %13, %8;\n\t" /* (a.d4 * a.d4).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4), "=r" (res->d5),
        "=r" (res->d6), "=r" (res->d7), "=r" (res->d8)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4));
}


__device__ static void mul_160_320_no_low5(int160 *res, int160 a, int160 b)
/*
res ~= a * b / 2^160
Carries into res.d0 are NOT computed. So the result differs from a full mul_160_320() / 2^160.
In a full mul_160_320() there are eight possible carries from res.d4 to res.d5. So ignoring the carries
the result is 0 to 8 lower than a full mul_160_320() / 2^160.
*/
{
  asm("{\n\t"
      "mul.hi.u32      %0, %5, %14;\n\t"     /* (a.d0 * b.d4).hi */

      "mad.lo.cc.u32   %0, %6, %14, %0;\n\t" /* (a.d1 * b.d4).lo */
      "addc.u32        %1, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %6, %13, %0;\n\t" /* (a.d1 * b.d3).hi */
      "madc.hi.u32     %1, %6, %14, %1;\n\t" /* (a.d1 * b.d4).hi */

      "mad.lo.cc.u32   %0, %7, %13, %0;\n\t" /* (a.d2 * b.d3).lo */
      "madc.lo.cc.u32  %1, %7, %14, %1;\n\t" /* (a.d2 * b.d4).lo */
      "addc.u32        %2, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %7, %12, %0;\n\t" /* (a.d2 * b.d2).hi */
      "madc.hi.cc.u32  %1, %7, %13, %1;\n\t" /* (a.d2 * b.d3).hi */
      "madc.hi.u32     %2, %7, %14, %2;\n\t" /* (a.d2 * b.d4).hi */

      "mad.lo.cc.u32   %0, %8, %12, %0;\n\t" /* (a.d3 * b.d2).lo */
      "madc.lo.cc.u32  %1, %8, %13, %1;\n\t" /* (a.d3 * b.d3).lo */
      "madc.lo.cc.u32  %2, %8, %14, %2;\n\t" /* (a.d3 * b.d4).lo */
      "addc.u32        %3, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %8, %11, %0;\n\t" /* (a.d3 * b.d1).hi */
      "madc.hi.cc.u32  %1, %8, %12, %1;\n\t" /* (a.d3 * b.d2).hi */
      "madc.hi.cc.u32  %2, %8, %13, %2;\n\t" /* (a.d3 * b.d3).hi */
      "madc.hi.u32     %3, %8, %14, %3;\n\t" /* (a.d3 * b.d4).hi */

      "mad.lo.cc.u32   %0, %9, %11, %0;\n\t" /* (a.d4 * b.d1).lo */
      "madc.lo.cc.u32  %1, %9, %12, %1;\n\t" /* (a.d4 * b.d2).lo */
      "madc.lo.cc.u32  %2, %9, %13, %2;\n\t" /* (a.d4 * b.d3).lo */
      "madc.lo.cc.u32  %3, %9, %14, %3;\n\t" /* (a.d4 * b.d4).lo */
      "addc.u32        %4, 0, 0;\n\t"

      "mad.hi.cc.u32   %0, %9, %10, %0;\n\t" /* (a.d4 * b.d0).hi */
      "madc.hi.cc.u32  %1, %9, %11, %1;\n\t" /* (a.d4 * b.d1).hi */
      "madc.hi.cc.u32  %2, %9, %12, %2;\n\t" /* (a.d4 * b.d2).hi */
      "madc.hi.cc.u32  %3, %9, %13, %3;\n\t" /* (a.d4 * b.d3).hi */
      "madc.hi.u32     %4, %9, %14, %4;\n\t" /* (a.d4 * b.d4).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4),
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3), "r" (b.d4));
}


__device__ static void mul_160(int160 *res, int160 a, int160 b)
/* res = a * b (only lower 160 bits of the result) */
{
  asm("{\n\t"
      "mul.lo.u32      %0, %5, %10;\n\t"     /* (a.d0 * b.d0).lo */
      "mul.lo.u32      %1, %5, %11;\n\t"     /* (a.d0 * b.d1).lo */
      "mul.lo.u32      %2, %5, %12;\n\t"     /* (a.d0 * b.d2).lo */
      "mul.lo.u32      %3, %5, %13;\n\t"     /* (a.d0 * b.d3).lo */
      "mul.lo.u32      %4, %5, %14;\n\t"     /* (a.d0 * b.d4).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t" /* (a.d0 * b.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t" /* (a.d0 * b.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t" /* (a.d0 * b.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t" /* (a.d0 * b.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t" /* (a.d1 * b.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t" /* (a.d1 * b.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t" /* (a.d1 * b.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t" /* (a.d1 * b.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t" /* (a.d1 * b.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t" /* (a.d1 * b.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t" /* (a.d1 * b.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t" /* (a.d2 * b.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t" /* (a.d2 * b.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t" /* (a.d2 * b.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t" /* (a.d2 * b.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t" /* (a.d2 * b.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t" /* (a.d3 * b.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t" /* (a.d3 * b.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t" /* (a.d3 * b.d0).hi */

      "mad.lo.u32      %4, %9, %10, %4;\n\t" /* (a.d4 * b.d0).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4)
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3), "r" (b.d4));
}


__device__ static void mulsub_160(int160 *res, int320 c, int160 a, int160 negb)
/* res = c - a * b (only lower 160 bits of the result) */
{
  asm("{\n\t"
      "mad.lo.cc.u32   %0, %5, %10, %15;\n\t" /* c += (a.d0 * negb.d0).lo */
      "madc.lo.cc.u32  %1, %5, %11, %16;\n\t" /* c += (a.d0 * negb.d1).lo */
      "madc.lo.cc.u32  %2, %5, %12, %17;\n\t" /* c += (a.d0 * negb.d2).lo */
      "madc.lo.cc.u32  %3, %5, %13, %18;\n\t" /* c += (a.d0 * negb.d3).lo */
      "madc.lo.u32     %4, %5, %14, %19;\n\t" /* c += (a.d0 * negb.d4).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t"  /* c += (a.d0 * negb.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t"  /* c += (a.d0 * negb.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t"  /* c += (a.d0 * negb.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t"  /* c += (a.d0 * negb.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t"  /* c += (a.d1 * negb.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t"  /* c += (a.d1 * negb.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t"  /* c += (a.d1 * negb.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t"  /* c += (a.d1 * negb.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t"  /* c += (a.d1 * negb.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t"  /* c += (a.d1 * negb.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t"  /* c += (a.d1 * negb.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t"  /* c += (a.d2 * negb.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t"  /* c += (a.d2 * negb.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t"  /* c += (a.d2 * negb.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t"  /* c += (a.d2 * negb.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t"  /* c += (a.d2 * negb.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t"  /* c += (a.d3 * negb.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t"  /* c += (a.d3 * negb.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t"  /* c += (a.d3 * negb.d0).hi */

      "mad.lo.u32      %4, %9, %10, %4;\n\t"  /* c += (a.d4 * negb.d0).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4),
        "r" (negb.d0), "r" (negb.d1), "r" (negb.d2), "r" (negb.d3), "r" (negb.d4),
        "r" (c.d0), "r" (c.d1), "r" (c.d2), "r" (c.d3), "r" (c.d4));
}


__device__ static void mulsub_160_288(int160 *res, int288 c, int160 a, int160 negb)
/* res = c - a * b (only lower 160 bits of the result) */
{
  asm("{\n\t"
      "mad.lo.cc.u32   %0, %5, %10, %15;\n\t" /* c += (a.d0 * negb.d0).lo */
      "madc.lo.cc.u32  %1, %5, %11, %16;\n\t" /* c += (a.d0 * negb.d1).lo */
      "madc.lo.cc.u32  %2, %5, %12, %17;\n\t" /* c += (a.d0 * negb.d2).lo */
      "madc.lo.cc.u32  %3, %5, %13, %18;\n\t" /* c += (a.d0 * negb.d3).lo */
      "madc.lo.u32     %4, %5, %14, %19;\n\t" /* c += (a.d0 * negb.d4).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t"  /* c += (a.d0 * negb.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t"  /* c += (a.d0 * negb.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t"  /* c += (a.d0 * negb.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t"  /* c += (a.d0 * negb.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t"  /* c += (a.d1 * negb.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t"  /* c += (a.d1 * negb.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t"  /* c += (a.d1 * negb.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t"  /* c += (a.d1 * negb.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t"  /* c += (a.d1 * negb.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t"  /* c += (a.d1 * negb.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t"  /* c += (a.d1 * negb.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t"  /* c += (a.d2 * negb.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t"  /* c += (a.d2 * negb.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t"  /* c += (a.d2 * negb.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t"  /* c += (a.d2 * negb.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t"  /* c += (a.d2 * negb.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t"  /* c += (a.d3 * negb.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t"  /* c += (a.d3 * negb.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t"  /* c += (a.d3 * negb.d0).hi */

      "mad.lo.u32      %4, %9, %10, %4;\n\t"  /* c += (a.d4 * negb.d0).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4),
        "r" (negb.d0), "r" (negb.d1), "r" (negb.d2), "r" (negb.d3), "r" (negb.d4),
        "r" (c.d0), "r" (c.d1), "r" (c.d2), "r" (c.d3), "r" (c.d4));
}


__device__ static void mul_160_128(int160 *res, int160 a, int128 b)
/* res = a * b (only lower 160 bits of the result) */
{
  asm("{\n\t"
      "mul.lo.u32      %0, %5, %10;\n\t"     /* (a.d0 * b.d0).lo */
      "mul.lo.u32      %1, %5, %11;\n\t"     /* (a.d0 * b.d1).lo */
      "mul.lo.u32      %2, %5, %12;\n\t"     /* (a.d0 * b.d2).lo */
      "mul.lo.u32      %3, %5, %13;\n\t"     /* (a.d0 * b.d3).lo */
      "mul.lo.u32      %4, %9, %10;\n\t"     /* (a.d4 * b.d0).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t" /* (a.d0 * b.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t" /* (a.d0 * b.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t" /* (a.d0 * b.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t" /* (a.d0 * b.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t" /* (a.d1 * b.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t" /* (a.d1 * b.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t" /* (a.d1 * b.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t" /* (a.d1 * b.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t" /* (a.d1 * b.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t" /* (a.d1 * b.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t" /* (a.d1 * b.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t" /* (a.d2 * b.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t" /* (a.d2 * b.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t" /* (a.d2 * b.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t" /* (a.d2 * b.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t" /* (a.d2 * b.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t" /* (a.d3 * b.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t" /* (a.d3 * b.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t" /* (a.d3 * b.d0).hi */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4)
        "r" (b.d0), "r" (b.d1), "r" (b.d2), "r" (b.d3));
}


__device__ static void mulsub_160_128_initial(int160 *res, int160 a, int128 negb)
/* res = 0 - a * b (only lower 160 bits of the result), negb.d4 = 0xFFFFFFFF */
{
  asm("{\n\t"
      "mul.lo.u32      %0, %5, %10;\n\t"      /* c += (a.d0 * negb.d0).lo */
      "mul.lo.u32      %1, %5, %11;\n\t"      /* c += (a.d0 * negb.d1).lo */
      "mul.lo.u32      %2, %5, %12;\n\t"      /* c += (a.d0 * negb.d2).lo */
      "mul.lo.u32      %3, %5, %13;\n\t"      /* c += (a.d0 * negb.d3).lo */
      "mul.lo.u32      %4, %9, %10;\n\t"      /* c += (a.d4 * negb.d0).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t"  /* c += (a.d0 * negb.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t"  /* c += (a.d0 * negb.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t"  /* c += (a.d0 * negb.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t"  /* c += (a.d0 * negb.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t"  /* c += (a.d1 * negb.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t"  /* c += (a.d1 * negb.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t"  /* c += (a.d1 * negb.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t"  /* c += (a.d1 * negb.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t"  /* c += (a.d1 * negb.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t"  /* c += (a.d1 * negb.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t"  /* c += (a.d1 * negb.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t"  /* c += (a.d2 * negb.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t"  /* c += (a.d2 * negb.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t"  /* c += (a.d2 * negb.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t"  /* c += (a.d2 * negb.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t"  /* c += (a.d2 * negb.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t"  /* c += (a.d3 * negb.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t"  /* c += (a.d3 * negb.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t"  /* c += (a.d3 * negb.d0).hi */

      "sub.u32         %4, %4, %5;\n\t"       /* c += (a.d0 * negb.d4).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4),
        "r" (negb.d0), "r" (negb.d1), "r" (negb.d2), "r" (negb.d3));
}


__device__ static void mulsub_160_128(int160 *res, int288 c, int160 a, int128 negb)
/* res = c - a * b (only lower 160 bits of the result), negb.d4 = 0xFFFFFFFF */
{
  asm("{\n\t"
      "mad.lo.cc.u32   %0, %5, %10, %14;\n\t" /* c += (a.d0 * negb.d0).lo */
      "madc.lo.cc.u32  %1, %5, %11, %15;\n\t" /* c += (a.d0 * negb.d1).lo */
      "madc.lo.cc.u32  %2, %5, %12, %16;\n\t" /* c += (a.d0 * negb.d2).lo */
      "madc.lo.cc.u32  %3, %5, %13, %17;\n\t" /* c += (a.d0 * negb.d3).lo */
      "madc.lo.u32     %4, %9, %10, %18;\n\t" /* c += (a.d4 * negb.d0).lo */

      "mad.hi.cc.u32   %1, %5, %10, %1;\n\t"  /* c += (a.d0 * negb.d0).hi */
      "madc.hi.cc.u32  %2, %5, %11, %2;\n\t"  /* c += (a.d0 * negb.d1).hi */
      "madc.hi.cc.u32  %3, %5, %12, %3;\n\t"  /* c += (a.d0 * negb.d2).hi */
      "madc.hi.u32     %4, %5, %13, %4;\n\t"  /* c += (a.d0 * negb.d3).hi */

      "mad.lo.cc.u32   %1, %6, %10, %1;\n\t"  /* c += (a.d1 * negb.d0).lo */
      "madc.lo.cc.u32  %2, %6, %11, %2;\n\t"  /* c += (a.d1 * negb.d1).lo */
      "madc.lo.cc.u32  %3, %6, %12, %3;\n\t"  /* c += (a.d1 * negb.d2).lo */
      "madc.lo.u32     %4, %6, %13, %4;\n\t"  /* c += (a.d1 * negb.d3).lo */

      "mad.hi.cc.u32   %2, %6, %10, %2;\n\t"  /* c += (a.d1 * negb.d0).hi */
      "madc.hi.cc.u32  %3, %6, %11, %3;\n\t"  /* c += (a.d1 * negb.d1).hi */
      "madc.hi.u32     %4, %6, %12, %4;\n\t"  /* c += (a.d1 * negb.d2).hi */

      "mad.lo.cc.u32   %2, %7, %10, %2;\n\t"  /* c += (a.d2 * negb.d0).lo */
      "madc.lo.cc.u32  %3, %7, %11, %3;\n\t"  /* c += (a.d2 * negb.d1).lo */
      "madc.lo.u32     %4, %7, %12, %4;\n\t"  /* c += (a.d2 * negb.d2).lo */

      "mad.hi.cc.u32   %3, %7, %10, %3;\n\t"  /* c += (a.d2 * negb.d0).hi */
      "madc.hi.u32     %4, %7, %11, %4;\n\t"  /* c += (a.d2 * negb.d1).hi */

      "mad.lo.cc.u32   %3, %8, %10, %3;\n\t"  /* c += (a.d3 * negb.d0).lo */
      "madc.lo.u32     %4, %8, %11, %4;\n\t"  /* c += (a.d3 * negb.d1).lo */

      "mad.hi.u32      %4, %8, %10, %4;\n\t"  /* c += (a.d3 * negb.d0).hi */

      "sub.u32         %4, %4, %5;\n\t"       /* c += (a.d0 * negb.d4).lo */
      "}"
      : "=r" (res->d0), "=r" (res->d1), "=r" (res->d2), "=r" (res->d3), "=r" (res->d4)
      : "r" (a.d0), "r" (a.d1), "r" (a.d2), "r" (a.d3), "r" (a.d4),
        "r" (negb.d0), "r" (negb.d1), "r" (negb.d2), "r" (negb.d3),
        "r" (c.d0), "r" (c.d1), "r" (c.d2), "r" (c.d3), "r" (c.d4));
}


__device__ static void div_352_160(int192 *res, int352 q, int160 n, float nf)
/* res = q / n (integer division) */
{
  float qf;
  unsigned int qi;
  int352 nn;
  int160 tmp160;

/********** Step 1, Offset 2^175 (5*32 + 15) **********/
  qf= __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);

//if ((qi >> 17) != 0) printf ("1/f fail 1\n");
//if (blockIdx.x==0 && threadIdx.x == 4) printf ("q1: %X\n", qi);
  qi <<= 15;
  res->d5 = qi;

// nn = n * qi
  nn.d5  =                                 __umul32(n.d0, qi);
  nn.d6  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d10 = __addc_cc(__umul32hi(n.d4, qi),                 0);

//  q = q - nn
  q.d5  = __sub_cc (q.d5, nn.d5);
  q.d6  = __subc_cc(q.d6, nn.d6);
  q.d7  = __subc_cc(q.d7, nn.d7);
  q.d8  = __subc_cc(q.d8, nn.d8);
  q.d9  = __subc_cc(q.d9, nn.d9);
  q.d10 = __subc   (q.d10, nn.d10);

/********** Step 2, Offset 2^155 (4*32 + 27) **********/
  qf= __uint2float_rn(q.d10);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 32.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 4) printf ("q2: %X\n", qi);

  res->d4 = qi << 27;
  res->d5 += qi >>  5;

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc   (__umul32hi(n.d4, qi),                  0);

// shiftleft nn 27 bits
  nn.d9  = (nn.d9  << 27) + (nn.d8  >> 5);
  nn.d8  = (nn.d8  << 27) + (nn.d7  >> 5);
  nn.d7  = (nn.d7  << 27) + (nn.d6  >> 5);
  nn.d6  = (nn.d6  << 27) + (nn.d5  >> 5);
  nn.d5  = (nn.d5  << 27) + (nn.d4  >> 5);
  nn.d4  =  nn.d4  << 27;

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc   (q.d9,  nn.d9);

/********** Step 3, Offset 2^135 (4*32 + 7) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 33554432.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 4) printf ("q3: %X\n", qi);

  qi <<= 7;
  res->d4 = __add_cc (res->d4, qi);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc   (q.d9,  nn.d9);

/********** Step 4, Offset 2^115 (3*32 + 19) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 8192.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==0 && threadIdx.x == 4) printf ("q4: %X\n", qi);

  res->d3  = qi << 19;
  res->d4 = __add_cc (res->d4, qi >> 13);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d3 =                                 __umul32(n.d0, qi);
  nn.d4 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d4, qi),                  0);

// shiftleft nn 19 bits
  nn.d8  = (nn.d8 << 19) + (nn.d7 >> 13);
  nn.d7  = (nn.d7 << 19) + (nn.d6 >> 13);
  nn.d6  = (nn.d6 << 19) + (nn.d5 >> 13);
  nn.d5  = (nn.d5 << 19) + (nn.d4 >> 13);
  nn.d4  = (nn.d4 << 19) + (nn.d3 >> 13);
  nn.d3  =  nn.d3 << 19;

//  q = q - nn
  q.d3  = __sub_cc (q.d3,  nn.d3);
  q.d4  = __subc_cc(q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc   (q.d8,  nn.d8);

/********** Step 5, Offset 2^95 (2*32 + 31) **********/
  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q5: %X\n", qi);

  res->d2 = qi << 31;
  res->d3 = __add_cc (res->d3, qi >> 1);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d7 >> 1 != q.d8) printf ("1/f fail 7\n");
// shiftleft nn 31 bits
  nn.d7 = (nn.d7 << 31) + (nn.d6 >> 1);
  nn.d6 = (nn.d6 << 31) + (nn.d5 >> 1);
  nn.d5 = (nn.d5 << 31) + (nn.d4 >> 1);
  nn.d4 = (nn.d4 << 31) + (nn.d3 >> 1);
  nn.d3 = (nn.d3 << 31) + (nn.d2 >> 1);
  nn.d2 =  nn.d2 << 31;

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 6, Offset 2^75 (2*32 + 11) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q6: %X\n", qi);

  qi <<= 11;
  res->d2 = __add_cc (res->d2, qi);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 7, Offset 2^55 (1*32 + 23) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q7: %X\n", qi);

  res->d1 = qi << 23;
  res->d2 = __add_cc (res->d2, qi >> 9);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d6 >> 9 != q.d7) printf ("1/f fail 7\n");
// shiftleft nn 23 bits
  nn.d6 = (nn.d6 << 23) + (nn.d5 >> 9);
  nn.d5 = (nn.d5 << 23) + (nn.d4 >> 9);
  nn.d4 = (nn.d4 << 23) + (nn.d3 >> 9);
  nn.d3 = (nn.d3 << 23) + (nn.d2 >> 9);
  nn.d2 = (nn.d2 << 23) + (nn.d1 >> 9);
  nn.d1 =  nn.d1 << 23;

// q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 8, Offset 2^35 (1*32 + 3) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q8: %X\n", qi);

  qi <<= 3;
  res->d1 = __add_cc (res->d1, qi);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 9, Offset 2^15 (0*32 + 15) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q9: %X\n", qi);

  res->d0 = qi << 15;
  res->d1 = __add_cc (res->d1, qi >> 17);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d0 =                                 __umul32(n.d0, qi);
  nn.d1 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc   (__umul32hi(n.d4, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d5 >> 17 != q.d6) printf ("1/f fail 9\n");

// shiftleft nn 15 bits
  nn.d5 = (nn.d5 << 15) + (nn.d4 >> 17);
  nn.d4 = (nn.d4 << 15) + (nn.d3 >> 17);
  nn.d3 = (nn.d3 << 15) + (nn.d2 >> 17);
  nn.d2 = (nn.d2 << 15) + (nn.d1 >> 17);
  nn.d1 = (nn.d1 << 15) + (nn.d0 >> 17);
  nn.d0 =  nn.d0 << 15;

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc   (q.d5, nn.d5);

/********** Step 10, Offset 2^0 (0*32 + 0) **********/

  qf= __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d3);

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q10: %X\n", qi);

  res->d0 = __add_cc (res->d0, qi);
  res->d1 = __addc_cc(res->d1, 0);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc_cc(res->d4, 0);
  res->d5 = __addc   (res->d5, 0);

// nn = n * qi
  nn.d0 =                                  __umul32(n.d0, qi);
  nn.d1 = __add_cc  (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc (__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc (__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc    (__umul32hi(n.d3, qi), __umul32(n.d4, qi));

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc   (q.d4, nn.d4);

/*
qi is always a little bit too small, this is OK for all steps except the last
one. Sometimes the result is a little bit bigger than n
*/

//if (blockIdx.x == 12 && threadIdx.x == 4)
//printf ("  rem: %X %X %X %X %X\r\n", q.d4, q.d3, q.d2, q.d1, q.d0);

  tmp160.d0 = q.d0;
  tmp160.d1 = q.d1;
  tmp160.d2 = q.d2;
  tmp160.d3 = q.d3;
  tmp160.d4 = q.d4;

  if(cmp_ge_160(tmp160,n))
  {
    res->d0 = __add_cc (res->d0, 1);
    res->d1 = __addc_cc(res->d1, 0);
    res->d2 = __addc_cc(res->d2, 0);
    res->d3 = __addc_cc(res->d3, 0);
    res->d4 = __addc_cc(res->d4, 0);
    res->d5 = __addc   (res->d5, 0);
  }
}


__device__ static void div_320_160(int160 *res, int320 q, int160 n, float nf)
/* res = q / n (integer division) */
{
  float qf;
  unsigned int qi;
  int320 nn;
  int160 tmp160;

/********** Step 2, Offset 2^155 (4*32 + 27) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 32.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q2: %X\n", qi);

  res->d4 = qi << 27;

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc   (__umul32hi(n.d4, qi),                  0);

// shiftleft nn 27 bits
  nn.d9  = (nn.d9  << 27) + (nn.d8  >> 5);
  nn.d8  = (nn.d8  << 27) + (nn.d7  >> 5);
  nn.d7  = (nn.d7  << 27) + (nn.d6  >> 5);
  nn.d6  = (nn.d6  << 27) + (nn.d5  >> 5);
  nn.d5  = (nn.d5  << 27) + (nn.d4  >> 5);
  nn.d4  =  nn.d4  << 27;

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc   (q.d9,  nn.d9);

/********** Step 3, Offset 2^135 (4*32 + 7) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf*= 33554432.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q3: %X\n", qi);

  qi <<= 7;
  res->d4 += qi;

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d9  = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc_cc(q.d8,  nn.d8);
  q.d9  = __subc   (q.d9,  nn.d9);

/********** Step 4, Offset 2^115 (3*32 + 19) **********/
  qf= __uint2float_rn(q.d9);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 8192.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q4: %X\n", qi);

  res->d3  = qi << 19;
  res->d4 += qi >> 13;

// nn = n * qi
  nn.d3 =                                 __umul32(n.d0, qi);
  nn.d4 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d4, qi),                  0);

// shiftleft nn 19 bits
  nn.d8  = (nn.d8 << 19) + (nn.d7 >> 13);
  nn.d7  = (nn.d7 << 19) + (nn.d6 >> 13);
  nn.d6  = (nn.d6 << 19) + (nn.d5 >> 13);
  nn.d5  = (nn.d5 << 19) + (nn.d4 >> 13);
  nn.d4  = (nn.d4 << 19) + (nn.d3 >> 13);
  nn.d3  =  nn.d3 << 19;

//  q = q - nn
  q.d3  = __sub_cc (q.d3,  nn.d3);
  q.d4  = __subc_cc(q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc   (q.d8,  nn.d8);

/********** Step 5, Offset 2^95 (2*32 + 31) **********/
  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q5: %X\n", qi);

  res->d2 = qi << 31;
  res->d3 = __add_cc (res->d3, qi >> 1);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d7 >> 1 != q.d8) printf ("1/f fail 7\n");
// shiftleft nn 31 bits
  nn.d7 = (nn.d7 << 31) + (nn.d6 >> 1);
  nn.d6 = (nn.d6 << 31) + (nn.d5 >> 1);
  nn.d5 = (nn.d5 << 31) + (nn.d4 >> 1);
  nn.d4 = (nn.d4 << 31) + (nn.d3 >> 1);
  nn.d3 = (nn.d3 << 31) + (nn.d2 >> 1);
  nn.d2 =  nn.d2 << 31;

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 6, Offset 2^75 (2*32 + 11) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q6: %X\n", qi);

  qi <<= 11;
  res->d2 = __add_cc (res->d2, qi);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 7, Offset 2^55 (1*32 + 23) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q7: %X\n", qi);

  res->d1 = qi << 23;
  res->d2 = __add_cc (res->d2, qi >> 9);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d6 >> 9 != q.d7) printf ("1/f fail 7\n");
// shiftleft nn 23 bits
  nn.d6 = (nn.d6 << 23) + (nn.d5 >> 9);
  nn.d5 = (nn.d5 << 23) + (nn.d4 >> 9);
  nn.d4 = (nn.d4 << 23) + (nn.d3 >> 9);
  nn.d3 = (nn.d3 << 23) + (nn.d2 >> 9);
  nn.d2 = (nn.d2 << 23) + (nn.d1 >> 9);
  nn.d1 =  nn.d1 << 23;

// q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 8, Offset 2^35 (1*32 + 3) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q8: %X\n", qi);

  qi <<= 3;
  res->d1 = __add_cc (res->d1, qi);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 9, Offset 2^15 (0*32 + 15) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q9: %X\n", qi);

  res->d0 = qi << 15;
  res->d1 = __add_cc (res->d1, qi >> 17);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d0 =                                 __umul32(n.d0, qi);
  nn.d1 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc   (__umul32hi(n.d4, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d5 >> 17 != q.d6) printf ("1/f fail 9\n");

// shiftleft nn 15 bits
  nn.d5 = (nn.d5 << 15) + (nn.d4 >> 17);
  nn.d4 = (nn.d4 << 15) + (nn.d3 >> 17);
  nn.d3 = (nn.d3 << 15) + (nn.d2 >> 17);
  nn.d2 = (nn.d2 << 15) + (nn.d1 >> 17);
  nn.d1 = (nn.d1 << 15) + (nn.d0 >> 17);
  nn.d0 =  nn.d0 << 15;

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc   (q.d5, nn.d5);

/********** Step 10, Offset 2^0 (0*32 + 0) **********/

  qf= __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d3);

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q10: %X\n", qi);

  res->d0 = __add_cc (res->d0, qi);
  res->d1 = __addc_cc(res->d1, 0);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d0 =                                  __umul32(n.d0, qi);
  nn.d1 = __add_cc  (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc (__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc (__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc    (__umul32hi(n.d3, qi), __umul32(n.d4, qi));

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc   (q.d4, nn.d4);

/*
qi is always a little bit too small, this is OK for all steps except the last
one. Sometimes the result is a little bit bigger than n
*/

//if (blockIdx.x == 12 && threadIdx.x == 4)
//printf ("  rem: %X %X %X %X %X\r\n", q.d4, q.d3, q.d2, q.d1, q.d0);

  tmp160.d0 = q.d0;
  tmp160.d1 = q.d1;
  tmp160.d2 = q.d2;
  tmp160.d3 = q.d3;
  tmp160.d4 = q.d4;

  if(cmp_ge_160(tmp160,n))
  {
    res->d0 = __add_cc (res->d0, 1);
    res->d1 = __addc_cc(res->d1, 0);
    res->d2 = __addc_cc(res->d2, 0);
    res->d3 = __addc_cc(res->d3, 0);
    res->d4 = __addc   (res->d4, 0);
  }
}


__device__ static void div_288_160(int160 *res, int288 q, int160 n, float nf)
/* res = q / n (integer division) */
{
  float qf;
  unsigned int qi;
  int288 nn;
  int160 tmp160;

/********** Step 2, Offset 2^155 (4*32 + 27) **********/
  qf= __uint2float_rn(q.d8);
  qf*= 32.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q2: %X\n", qi);

  res->d4 = qi << 27;

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));

// shiftleft nn 27 bits
  nn.d8  = (nn.d8  << 27) + (nn.d7  >> 5);
  nn.d7  = (nn.d7  << 27) + (nn.d6  >> 5);
  nn.d6  = (nn.d6  << 27) + (nn.d5  >> 5);
  nn.d5  = (nn.d5  << 27) + (nn.d4  >> 5);
  nn.d4  =  nn.d4  << 27;

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc   (q.d8,  nn.d8);

/********** Step 3, Offset 2^135 (4*32 + 7) **********/
  qf= __uint2float_rn(q.d8);
  qf*= 33554432.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q3: %X\n", qi);

  qi <<= 7;
  res->d4 += qi;

// nn = n * qi
  nn.d4  =                                 __umul32(n.d0, qi);
  nn.d5  = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d6  = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d7  = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d8  = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));

//  q = q - nn
  q.d4  = __sub_cc (q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc   (q.d8,  nn.d8);

/********** Step 4, Offset 2^115 (3*32 + 19) **********/
  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf*= 8192.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q4: %X\n", qi);

  res->d3  = qi << 19;
  res->d4 += qi >> 13;

// nn = n * qi
  nn.d3 =                                 __umul32(n.d0, qi);
  nn.d4 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d8 = __addc_cc(__umul32hi(n.d4, qi),                  0);

// shiftleft nn 19 bits
  nn.d8  = (nn.d8 << 19) + (nn.d7 >> 13);
  nn.d7  = (nn.d7 << 19) + (nn.d6 >> 13);
  nn.d6  = (nn.d6 << 19) + (nn.d5 >> 13);
  nn.d5  = (nn.d5 << 19) + (nn.d4 >> 13);
  nn.d4  = (nn.d4 << 19) + (nn.d3 >> 13);
  nn.d3  =  nn.d3 << 19;

//  q = q - nn
  q.d3  = __sub_cc (q.d3,  nn.d3);
  q.d4  = __subc_cc(q.d4,  nn.d4);
  q.d5  = __subc_cc(q.d5,  nn.d5);
  q.d6  = __subc_cc(q.d6,  nn.d6);
  q.d7  = __subc_cc(q.d7,  nn.d7);
  q.d8  = __subc   (q.d8,  nn.d8);

/********** Step 5, Offset 2^95 (2*32 + 31) **********/
  qf= __uint2float_rn(q.d8);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q5: %X\n", qi);

  res->d2 = qi << 31;
  res->d3 = __add_cc (res->d3, qi >> 1);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d7 >> 1 != q.d8) printf ("1/f fail 7\n");
// shiftleft nn 31 bits
  nn.d7 = (nn.d7 << 31) + (nn.d6 >> 1);
  nn.d6 = (nn.d6 << 31) + (nn.d5 >> 1);
  nn.d5 = (nn.d5 << 31) + (nn.d4 >> 1);
  nn.d4 = (nn.d4 << 31) + (nn.d3 >> 1);
  nn.d3 = (nn.d3 << 31) + (nn.d2 >> 1);
  nn.d2 =  nn.d2 << 31;

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 6, Offset 2^75 (2*32 + 11) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf*= 2097152.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q6: %X\n", qi);

  qi <<= 11;
  res->d2 = __add_cc (res->d2, qi);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d2 =                                 __umul32(n.d0, qi);
  nn.d3 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d7 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d2 = __sub_cc (q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc_cc(q.d6, nn.d6);
  q.d7 = __subc   (q.d7, nn.d7);

/********** Step 7, Offset 2^55 (1*32 + 23) **********/
  qf= __uint2float_rn(q.d7);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 512.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q7: %X\n", qi);

  res->d1 = qi << 23;
  res->d2 = __add_cc (res->d2, qi >> 9);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//if (nn.d6 >> 9 != q.d7) printf ("1/f fail 7\n");
// shiftleft nn 23 bits
  nn.d6 = (nn.d6 << 23) + (nn.d5 >> 9);
  nn.d5 = (nn.d5 << 23) + (nn.d4 >> 9);
  nn.d4 = (nn.d4 << 23) + (nn.d3 >> 9);
  nn.d3 = (nn.d3 << 23) + (nn.d2 >> 9);
  nn.d2 = (nn.d2 << 23) + (nn.d1 >> 9);
  nn.d1 =  nn.d1 << 23;

// q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 8, Offset 2^35 (1*32 + 3) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf*= 536870912.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q8: %X\n", qi);

  qi <<= 3;
  res->d1 = __add_cc (res->d1, qi);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d1 =                                 __umul32(n.d0, qi);
  nn.d2 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d5 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d6 = __addc_cc(__umul32hi(n.d4, qi),                  0);

//  q = q - nn
  q.d1 = __sub_cc (q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc_cc(q.d5, nn.d5);
  q.d6 = __subc   (q.d6, nn.d6);

/********** Step 9, Offset 2^15 (0*32 + 15) **********/

  qf= __uint2float_rn(q.d6);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf*= 131072.0f;

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q9: %X\n", qi);

  res->d0 = qi << 15;
  res->d1 = __add_cc (res->d1, qi >> 17);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d0 =                                 __umul32(n.d0, qi);
  nn.d1 = __add_cc (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc(__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc(__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc_cc(__umul32hi(n.d3, qi), __umul32(n.d4, qi));
  nn.d5 = __addc   (__umul32hi(n.d4, qi),                  0);

//if (blockIdx.x==12 && threadIdx.x == 4) if (nn.d5 >> 17 != q.d6) printf ("1/f fail 9\n");

// shiftleft nn 15 bits
  nn.d5 = (nn.d5 << 15) + (nn.d4 >> 17);
  nn.d4 = (nn.d4 << 15) + (nn.d3 >> 17);
  nn.d3 = (nn.d3 << 15) + (nn.d2 >> 17);
  nn.d2 = (nn.d2 << 15) + (nn.d1 >> 17);
  nn.d1 = (nn.d1 << 15) + (nn.d0 >> 17);
  nn.d0 =  nn.d0 << 15;

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc_cc(q.d4, nn.d4);
  q.d5 = __subc   (q.d5, nn.d5);

/********** Step 10, Offset 2^0 (0*32 + 0) **********/

  qf= __uint2float_rn(q.d5);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf= qf * 4294967296.0f + __uint2float_rn(q.d3);

  qi=__float2uint_rz(qf*nf);
//if (blockIdx.x==12 && threadIdx.x == 4) printf ("q10: %X\n", qi);

  res->d0 = __add_cc (res->d0, qi);
  res->d1 = __addc_cc(res->d1, 0);
  res->d2 = __addc_cc(res->d2, 0);
  res->d3 = __addc_cc(res->d3, 0);
  res->d4 = __addc   (res->d4, 0);

// nn = n * qi
  nn.d0 =                                  __umul32(n.d0, qi);
  nn.d1 = __add_cc  (__umul32hi(n.d0, qi), __umul32(n.d1, qi));
  nn.d2 = __addc_cc (__umul32hi(n.d1, qi), __umul32(n.d2, qi));
  nn.d3 = __addc_cc (__umul32hi(n.d2, qi), __umul32(n.d3, qi));
  nn.d4 = __addc    (__umul32hi(n.d3, qi), __umul32(n.d4, qi));

//  q = q - nn
  q.d0 = __sub_cc (q.d0, nn.d0);
  q.d1 = __subc_cc(q.d1, nn.d1);
  q.d2 = __subc_cc(q.d2, nn.d2);
  q.d3 = __subc_cc(q.d3, nn.d3);
  q.d4 = __subc   (q.d4, nn.d4);

/*
qi is always a little bit too small, this is OK for all steps except the last
one. Sometimes the result is a little bit bigger than n
*/

//if (blockIdx.x == 12 && threadIdx.x == 4)
//printf ("  rem: %X %X %X %X %X\r\n", q.d4, q.d3, q.d2, q.d1, q.d0);

  tmp160.d0 = q.d0;
  tmp160.d1 = q.d1;
  tmp160.d2 = q.d2;
  tmp160.d3 = q.d3;
  tmp160.d4 = q.d4;

  if(cmp_ge_160(tmp160,n))
  {
    res->d0 = __add_cc (res->d0, 1);
    res->d1 = __addc_cc(res->d1, 0);
    res->d2 = __addc_cc(res->d2, 0);
    res->d3 = __addc_cc(res->d3, 0);
    res->d4 = __addc   (res->d4, 0);
  }
}


__device__ static void mod_simple_192_160(int160 *res, int192 q, int160 n, float nf)
/*
res = q mod n
used for refinement in barrett modular multiplication
assumes q < Xn where X is a small integer
*/
{
  float qf;
  unsigned int qi;
  int192 nn;

  qf = __uint2float_rn(q.d5);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d4);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d3);

  qi=__float2uint_rz(qf*nf);

  nn.d0 =                           __umul32(n.d0, qi);
  nn.d1 = __umad32hi_cc  (n.d0, qi, __umul32(n.d1, qi));
  nn.d2 = __umad32hic_cc (n.d1, qi, __umul32(n.d2, qi));
  nn.d3 = __umad32hic_cc (n.d2, qi, __umul32(n.d3, qi));
  nn.d4 = __umad32hic_cc (n.d3, qi, __umul32(n.d4, qi));
  nn.d5 = __umad32hic    (n.d4, qi,                  0);

  res->d0 = __sub_cc (q.d0, nn.d0);
  res->d1 = __subc_cc(q.d1, nn.d1);
  res->d2 = __subc_cc(q.d2, nn.d2);
  res->d3 = __subc_cc(q.d3, nn.d3);
  res->d4 = __subc_cc(q.d4, nn.d4);
  q.d5    = __subc   (q.d5, nn.d5);

  if(q.d5 || cmp_ge_160(*res, n))		// final adjustment in case finalrem >= f
  {
    sub_160(res, *res, n);
  }
}


__device__ static void mod_simple_160(int160 *res, int160 q, int160 n, float nf)
/*
res = q mod n
used for refinement in barrett modular multiplication
assumes q < Xn where X is a small integer
*/
{
  float qf;
  unsigned int qi;
  int160 nn;

  qf = __uint2float_rn(q.d4);
  qf = qf * 4294967296.0f + __uint2float_rn(q.d3);

  qi=__float2uint_rz(qf*nf);

  nn.d0 =                           __umul32(n.d0, qi);
  nn.d1 = __umad32hi_cc  (n.d0, qi, __umul32(n.d1, qi));
  nn.d2 = __umad32hic_cc (n.d1, qi, __umul32(n.d2, qi));
  nn.d3 = __umad32hic_cc (n.d2, qi, __umul32(n.d3, qi));
  nn.d4 = __umad32hic    (n.d3, qi, __umul32(n.d4, qi));

  sub_160(res, q, nn);

  if(cmp_ge_160(*res, n))			// final adjustment in case finalrem >= f
  {
    sub_160(res, *res, n);
  }
}
