/*
This file is part of mfaktc.
Copyright (C) 2009, 2010, 2011, 2012  Oliver Weihe (o.weihe@t-online.de)

mfaktc is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

mfaktc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                
You should have received a copy of the GNU General Public License
along with mfaktc.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "compatibility.h"
#include "params.h"

unsigned long long int calculate_k_min(int fermat, unsigned int exp, int bits);	// From mfaktc.c
unsigned long long int calculate_k_max(int fermat, unsigned int exp, int bits);

unsigned int checkpoint_checksum(char *string, int chars)
/* generates a CRC-32 like checksum of the string */
{
  unsigned int chksum=0;
  int i,j;
  
  for(i=0;i<chars;i++)
  {
    for(j=7;j>=0;j--)
    {
      if((chksum>>31) == (((unsigned int)(string[i]>>j))&1))
      {
        chksum<<=1;
      }
      else
      {
        chksum = (chksum<<1)^0x04C11DB7;
      }
    }
  }
  return chksum;
}

void checkpoint_write(int fermat, unsigned int exp, int bit_min, int bit_max, unsigned long k_lower_bound, unsigned long k_upper_bound, int cur_class, int num_factors)
/*
checkpoint_write() writes the checkpoint file.
*/
{
  FILE *f;
  char buffer[100], filename[20];
  unsigned long long k_min, k_max;
  unsigned int i;

  k_min=calculate_k_min(fermat, exp, bit_min);
  k_max=calculate_k_max(fermat, exp, bit_max);
  if (k_lower_bound > k_min) k_min = k_lower_bound;
  if (k_upper_bound && k_upper_bound < k_max) k_max = k_upper_bound;

  sprintf(filename, "%s%u.ckp", fermat ? "F" : "M", exp);
  
  f=fopen(filename, "w");
  if(f==NULL)
  {
    printf("WARNING, could not write checkpoint file \"%s\"\n", filename);
  }
  else
  {
    sprintf(buffer,"%u %d %d %" PRIu64 " %" PRIu64 " %d %s: %d %d", exp, bit_min, bit_max, k_min, k_max, NUM_CLASSES, MFAKTC_VERSION, cur_class, num_factors);
    i=checkpoint_checksum(buffer,strlen(buffer));
    fprintf(f,"%s %08X", buffer, i);
    fclose(f);
  }
}


int checkpoint_read(int fermat, unsigned int exp, int bit_min, int bit_max, unsigned long k_lower_bound, unsigned long k_upper_bound, int dont_checksum, int *cur_class, int *num_factors)
/*
checkpoint_read() reads the checkpoint file and compares values for exp,
bit_min, bit_max, NUM_CLASSES read from file with current values.
If these parameters are equal than it sets cur_class and num_factors to the
values from the checkpoint file.

returns 1 on success (valid checkpoint file)
returns 0 otherwise
*/
{
  FILE *f;
  int ret=0,i,chksum;
  char buffer[100], buffer2[100], *ptr, filename[20];
  unsigned long long k_min, k_max;

  k_min=calculate_k_min(fermat, exp, bit_min);
  k_max=calculate_k_max(fermat, exp, bit_max);
  if (k_lower_bound > k_min) k_min = k_lower_bound;
  if (k_upper_bound && k_upper_bound < k_max) k_max = k_upper_bound;

  for(i=0;i<100;i++)buffer[i]=0;

  *cur_class=-1;
  *num_factors=0;
  
  sprintf(filename, "%s%u.ckp", fermat ? "F" : "M", exp);
  
  f=fopen(filename, "r");
  if(f==NULL)
  {
    return 0;
  }
  i=fread(buffer,sizeof(char),99,f);
  sprintf(buffer2,"%u %d %d %" PRIu64 " %" PRIu64 " %d %s:", exp, bit_min, bit_max, k_min, k_max, NUM_CLASSES, MFAKTC_VERSION);
  ptr=strstr(buffer, buffer2);
  if(ptr==buffer)
  {
    i=strlen(buffer2);
    if(i<70)
    {
      ptr=&(buffer[i]);
      sscanf(ptr,"%d %d", cur_class, num_factors);
      sprintf(buffer2,"%s %d %d", buffer2, *cur_class, *num_factors);
      chksum=checkpoint_checksum(buffer2,strlen(buffer2));
      sprintf(buffer2,"%s %08X", buffer2, chksum);
      if(*cur_class >= 0 && \
         *cur_class < NUM_CLASSES && \
	 *num_factors >= 0 &&
	 (dont_checksum || (strlen(buffer) == strlen(buffer2) && strstr(buffer, buffer2) == buffer)))
      {
        ret=1;
      }
    }
  }
  fclose(f);
  return ret;
}


void checkpoint_delete(int fermat, unsigned int exp)
/*
tries to delete the checkpoint file
*/
{
  char filename[20];
  sprintf(filename, "%s%u.ckp", fermat ? "F" : "M", exp);
  
  if(remove(filename))
  {
    if(errno != ENOENT) /* ENOENT = "No such file or directory" -> there was no checkpoint file */
    {
      printf("WARNING: can't delete the checkpoint file \"%s\"\n", filename);
    }
  }
}
