#include <gmp.h>
#include <stdio.h>
#include <string.h>
#include <inttypes.h>
#include <time.h>
#include <stdbool.h>

bool    stripCRLF(char *line);
int64_t countKInFile(char *fileName);

bool    ib_isABC;
int32_t ii_n;

int main(int argc, char **argv)
{
   FILE    *inPtr;
   FILE    *dmPtr;
   char     buffer[1000];
   int64_t  kCount;
   uint64_t k, kTested;
   uint32_t ignore, factorsFound = 0;
   double   kTestedPerSecond, kSecondsPerTest;
   double   percentCompleted;
   time_t   startTime, lastTime, currentTime, estimatedFinishTime;
   char     finishTimeBuffer[40];
   struct tm   *finish_tm;
   mpz_t    rem, mersenne, nTemp, kTemp, factor;

   if (argc == 1)
   {
      printf("Provide an input file name on the command line\n");
      return 4;
   }

   ib_isABC = false;
   kCount = countKInFile(argv[1]);
   
   if (kCount < 0)
      return 4;

   inPtr = fopen(argv[1], "r");
   startTime = lastTime = time(NULL);

   mpz_init(rem);
   mpz_init(mersenne);
   mpz_init(nTemp);
   mpz_init(kTemp);
   mpz_init(factor);

   mpz_set_ui(nTemp, 2);
   mpz_pow_ui(nTemp, nTemp, ii_n);
   mpz_sub_ui(nTemp, nTemp, 1);
   mpz_set_ui(mersenne, 2);
   printf("Testing %u terms to see if they divide 2^(2^%u-1)-1\n", kCount, ii_n);

   while (fgets(buffer, sizeof(buffer), inPtr) != NULL)
   {
      if (!stripCRLF(buffer))
         continue;

      if (buffer[0] == 'A')
         continue;

      if (ib_isABC)
      {
         if (sscanf(buffer, "%" SCNu64"", &k) != 1)
         {
            printf("Could not parse line %s from input file\n", buffer);
            return 4;
         }
      }
      else if (sscanf(buffer, "2*%llu*(2^%u-1)+1", &k, &ignore) != 2)
      {
         printf("Could not parse line %s from input file\n", buffer);
         return 4;
      }
      
      kTested++;

#ifdef WIN32
      // Even though built with 64-bit limbs, mpz_set_ui doesn't
      // populate kTemp correctly when k > 32 bits.
      mpz_set_ui(kTemp, k >> 32);
      mpz_mul_2exp(kTemp, kTemp, 32);
      mpz_add_ui(kTemp, kTemp, k & (0xffffffff));
#else
      mpz_set_ui(kTemp, k);
#endif

      mpz_mul(factor, kTemp, nTemp);
      mpz_mul_ui(factor, factor, 2);
      mpz_add_ui(factor, factor, 1);

      mpz_powm(rem, mersenne, nTemp, factor);

      if (mpz_cmp_ui(rem, 1) == 0)
      {
         factorsFound++;
         
         dmPtr = fopen("dm_factors.txt", "a+");
         printf("Found factor 2*%" PRIu64"*(2^%u-1)+1 of 2^(2^%u-1)-1\n", k, ii_n, ii_n);
         fprintf(dmPtr, "Found factor 2*%" PRIu64"*(2^%u-1)+1 of 2^(2^%u-1)-1\n", k, ii_n, ii_n);
         fclose(dmPtr);
      }

      currentTime = time(NULL);

      if (currentTime > lastTime + 60)
      {
         percentCompleted = (double) kTested / (double) kCount;
         
         kTestedPerSecond = (double) kTested / (double) (currentTime - startTime);
         
         estimatedFinishTime = startTime + (currentTime - startTime)/percentCompleted;
         finish_tm = localtime(&estimatedFinishTime);
         strftime(finishTimeBuffer, sizeof(finishTimeBuffer), "%Y-%m-%d %H:%M", finish_tm);
         
         if (kTestedPerSecond >= 1.0)
            printf(" Tested %5.2f pct of range at %.2lf tests per second  ETC %s\r", 
                           percentCompleted * 100.0, kTestedPerSecond, finishTimeBuffer);
         else 
         {
            kSecondsPerTest = (double) (currentTime - startTime) / (double) kTested;

            printf(" Tested %5.2f pct of range at %.2lf seconds per test ETC %s\r", 
                           percentCompleted * 100.0, kSecondsPerTest, finishTimeBuffer);
         }

         lastTime = currentTime;
      }
   }

   fclose(inPtr);

   
   if (currentTime > startTime)
   {
      kTestedPerSecond = (double) kTested / (double) (currentTime - startTime);
   
      if (kTestedPerSecond >= 1)
         printf("Tested %" PRIu64" terms for divisilibity at %.2lf tests per second\n", kTested, kTestedPerSecond);
      else 
      {
         kSecondsPerTest = (double) (currentTime - startTime) / (double) kTested;
               
         printf("Tested %" PRIu64" terms for divisilibity at %.2lf seconds per test\n", kTested, kSecondsPerTest);
      }
   }
   
   mpz_clear(rem);
   mpz_clear(mersenne);
   mpz_clear(nTemp);
   mpz_clear(kTemp);
   mpz_clear(factor);

   currentTime = time(NULL);

   printf("Double-Mersenne divisibility checking completed in %llu seconds.  %u terms tested.   %u factors found",
      (currentTime - startTime), kTested, factorsFound);
   
   return 0;
}

int64_t countKInFile(char *fileName)
{
   FILE    *inPtr = fopen(fileName, "r");
   int64_t  kCount = 0;
   uint64_t k;
   uint32_t n, thisN;
   char     buffer[1000];

   if (!inPtr)
   {
      printf("Could not open file %s\n", fileName);
      return -1;
   }

   while (fgets(buffer, sizeof(buffer), inPtr) != NULL)
   {
      if (!stripCRLF(buffer))
         continue;

      if (sscanf(buffer, "ABC 2*$a*(2^%u-1)+1", &thisN) == 1)
      {
         ib_isABC = true;
         continue;
      }

      if (ib_isABC)
      {
         if (sscanf(buffer, "%" SCNu64"", &k) != 1)
         {
            printf("Could not parse line %s from input file\n", buffer);
            return -1;
         }
      }
      else if (sscanf(buffer, "2*%llu*(2^%u-1)+1", &k, &thisN) != 2)
      {
         printf("Could not parse line %s from input file\n", buffer);
         return -1;
      }
    
      if (ii_n == 0)
         ii_n = thisN;
      
      if (thisN != ii_n)
      {
         printf("Input file has mixed n\n");
         return -1;
      }
    
      kCount++;
   }

   fclose(inPtr);
   
   return kCount;
}

bool stripCRLF(char *line)
{
   uint32_t n;
   
   n = strlen(line) - 1;
   while (n >= 0 && (line[n] == '\n' || line[n] == '\r'))
   {
      line[n] = 0;
      
      // If the length of the line is 0, return false
      if (n == 0)
         return false;
      
      n--;
   }

   return true;
}
